#ifndef _EFL_UI_WIDGET_EO_LEGACY_H_
#define _EFL_UI_WIDGET_EO_LEGACY_H_

#ifndef _EFL_UI_WIDGET_EO_CLASS_TYPE
#define _EFL_UI_WIDGET_EO_CLASS_TYPE

typedef Eo Efl_Ui_Widget;

#endif

#ifndef _EFL_UI_WIDGET_EO_TYPES
#define _EFL_UI_WIDGET_EO_TYPES

/** Function pointer for on show region hook
 *
 * @ingroup Efl_Ui
 */
typedef void (*Efl_Ui_Scrollable_On_Show_Region)(void *data, Efl_Canvas_Object *obj, Eina_Rect region);

/** All relevant fields needed for the current state of focus registeration
 *
 * @ingroup Efl_Ui
 */
typedef struct _Efl_Ui_Widget_Focus_State
{
  Efl_Ui_Focus_Manager *manager; /**< The manager where the widget is registered
                                  * in */
  Efl_Ui_Focus_Object *parent; /**< The parent the widget is using as logical
                                * parent */
  Eina_Bool logical; /**< @c true if this is registered as logical currently */
} Efl_Ui_Widget_Focus_State;


#endif

/**
 * @brief This is the internal canvas object managed by a widget.
 *
 * This property is protected as it is meant for widget implementations only,
 * to set and access the internal canvas object. Do use this function unless
 * you're implementing a widget.
 *
 * Sets the new resize object for this widget.
 *
 * @param[in] obj The object.
 * @param[in] sobj A canvas object (often a @ref Efl_Canvas_Layout object).
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_resize_object_set(Efl_Ui_Widget *obj, Efl_Canvas_Object *sobj);

/**
 * @brief Whether the widget is enabled (accepts and reacts to user inputs).
 *
 * Each widget may handle the disabled state differently, but overall disabled
 * widgets shall not respond to any input events. This is @c false by default,
 * meaning the widget is enabled.
 *
 * Enables or disables this widget.
 *
 * Disabling a widget will disable all its children recursively, but only this
 * widget will be marked as disabled internally.
 *
 * @param[in] obj The object.
 * @param[in] disabled @c true if the widget is disabled.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_disabled_set(Efl_Ui_Widget *obj, Eina_Bool disabled);

/**
 * @brief Whether the widget is enabled (accepts and reacts to user inputs).
 *
 * Each widget may handle the disabled state differently, but overall disabled
 * widgets shall not respond to any input events. This is @c false by default,
 * meaning the widget is enabled.
 *
 * Returns whether the widget is disabled.
 *
 * This will return @c true if any widget in the parent hierarchy is disabled.
 * Re-enabling that parent may in turn change the disabled state of this
 * widget.
 *
 * @param[in] obj The object.
 *
 * @return @c true if the widget is disabled.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Eina_Bool elm_widget_disabled_get(const Efl_Ui_Widget *obj);

/**
 * @brief The widget style to use.
 *
 * Styles define different look and feel for widgets, and may provide different
 * parts for layout-based widgets. Styles vary from widget to widget and may be
 * defined by other themes by means of extensions and overlays.
 *
 * The style can only be set before @ref Efl.Object.finalize, which means at
 * construction time of the object (inside @c efl_add in C).
 *
 * Can only be called during construction, before finalize.
 *
 * @param[in] obj The object.
 * @param[in] style Name of the style to use. Refer to each widget's
 * documentation for the available style names, or to the themes in use.
 *
 * @return Whether the style was successfully applied or not, see the values of
 * @ref Efl_Ui_Theme_Apply for more information.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Efl_Ui_Theme_Apply elm_widget_style_set(Efl_Ui_Widget *obj, const char *style);

/**
 * @brief The widget style to use.
 *
 * Styles define different look and feel for widgets, and may provide different
 * parts for layout-based widgets. Styles vary from widget to widget and may be
 * defined by other themes by means of extensions and overlays.
 *
 * The style can only be set before @ref Efl.Object.finalize, which means at
 * construction time of the object (inside @c efl_add in C).
 *
 * Returns the current style of a widget.
 *
 * @param[in] obj The object.
 *
 * @return Name of the style to use. Refer to each widget's documentation for
 * the available style names, or to the themes in use.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI const char *elm_widget_style_get(const Efl_Ui_Widget *obj);


/**
 * @brief The ability for a widget to be focused.
 *
 * Unfocusable objects do nothing when programmatically focused. The nearest
 * focusable parent object the one really getting focus. Also, when they
 * receive mouse input, they will get the event, but not take away the focus
 * from where it was previously.
 *
 * @note Objects which are meant to be interacted with by input events are
 * created able to be focused, by default. All the others are not.
 *
 * This property's default value depends on the widget (eg. a box is not
 * focusable, but a button is).
 *
 * @param[in] obj The object.
 * @param[in] can_focus Whether the object is focusable.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_can_focus_set(Efl_Ui_Widget *obj, Eina_Bool can_focus);

/**
 * @brief The ability for a widget to be focused.
 *
 * Unfocusable objects do nothing when programmatically focused. The nearest
 * focusable parent object the one really getting focus. Also, when they
 * receive mouse input, they will get the event, but not take away the focus
 * from where it was previously.
 *
 * @note Objects which are meant to be interacted with by input events are
 * created able to be focused, by default. All the others are not.
 *
 * This property's default value depends on the widget (eg. a box is not
 * focusable, but a button is).
 *
 * @param[in] obj The object.
 *
 * @return Whether the object is focusable.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Eina_Bool elm_widget_can_focus_get(const Efl_Ui_Widget *obj);

/**
 * @brief The internal parent of this widget.
 *
 * @ref Efl_Ui_Widget objects have a parent hierarchy that may differ slightly
 * from their @ref Efl_Object or @ref Efl_Canvas_Object hierarchy. This is
 * meant for internal handling.
 *
 * See also @ref elm_widget_top_get.
 *
 * @param[in] obj The object.
 * @param[in] parent Widget parent object
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_widget_parent_set(Efl_Ui_Widget *obj, Efl_Ui_Widget *parent);

/**
 * @brief The internal parent of this widget.
 *
 * @ref Efl_Ui_Widget objects have a parent hierarchy that may differ slightly
 * from their @ref Efl_Object or @ref Efl_Canvas_Object hierarchy. This is
 * meant for internal handling.
 *
 * See also @ref elm_widget_top_get.
 *
 * @param[in] obj The object.
 *
 * @return Widget parent object
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Efl_Ui_Widget *elm_widget_parent_get(const Efl_Ui_Widget *obj);

/**
 * @brief Root widget in the widget hierarchy.
 *
 * This returns the top widget, in terms of widget hierarchy. This is usually a
 * window (@Efl.Ui.Win). This function walks the list of
 * @ref elm_widget_parent_get.
 *
 * If this widget has no parent (in terms of widget hierarchy) this will return
 * @c null.
 *
 * @note This may not be a display manager window in case of nested canvases.
 * If a "real" window is required, then you might want to verify that the
 * returned object is a @c Efl.Ui.Win_Inlined, and then get
 * @c Efl.Ui.Win_Inlined.inlined_parent to find an object in the master window.
 *
 * See also @ref elm_widget_parent_get.
 *
 * @param[in] obj The object.
 *
 * @return Top widget, usually a window.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Efl_Ui_Widget *elm_widget_top_get(const Efl_Ui_Widget *obj);

/**
 * @brief Accessibility information.
 *
 * This is a replacement string to be read by the accessibility text-to-speech
 * engine, if accessibility is enabled by configuration. This will take
 * precedence over the default text for this object, which means for instance
 * that the label of a button won't be read out loud, instead @c txt will be
 * read out.
 *
 * @param[in] obj The object.
 * @param[in] txt Accessibility text description.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_access_info_set(Efl_Ui_Widget *obj, const char *txt);

/**
 * @brief Accessibility information.
 *
 * This is a replacement string to be read by the accessibility text-to-speech
 * engine, if accessibility is enabled by configuration. This will take
 * precedence over the default text for this object, which means for instance
 * that the label of a button won't be read out loud, instead @c txt will be
 * read out.
 *
 * @param[in] obj The object.
 *
 * @return Accessibility text description.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI const char *elm_widget_access_info_get(const Efl_Ui_Widget *obj);

/**
 * @brief Virtual function handling sub objects being added.
 *
 * Sub objects can be any canvas object, not necessarily widgets.
 *
 * See also @ref elm_widget_parent_get.
 *
 * @param[in] obj The object.
 * @param[in] sub_obj Sub object to be added. Not necessarily a widget itself.
 *
 * @return Indicates if the operation succeeded.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Eina_Bool elm_widget_sub_object_add(Efl_Ui_Widget *obj, Efl_Canvas_Object *sub_obj);

/**
 * @brief Virtual function handling sub objects being removed.
 *
 * Sub objects can be any canvas object, not necessarily widgets.
 *
 * See also @ref elm_widget_parent_get.
 *
 * @param[in] obj The object.
 * @param[in] sub_obj Sub object to be removed. Should be a child of this
 * widget.
 *
 * @return Indicates if the operation succeeded.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Eina_Bool elm_widget_sub_object_del(Efl_Ui_Widget *obj, Efl_Canvas_Object *sub_obj);

/**
 * @brief Whether the widget's automatic orientation is enabled or not.
 *
 * Orientation mode is used for widgets to change their style or send signals
 * based on the canvas rotation (i.e. the window orientation). If the
 * orientation mode is enabled, the widget will emit signals such as
 * "elm,state,orient,N" where @c N is one of 0, 90, 180, 270, depending on the
 * window orientation. Such signals may be handled by the theme in order to
 * provide a different look for the widget based on the canvas orientation.
 *
 * By default orientation mode is enabled.
 *
 * See also @ref elm_widget_on_orientation_update.
 *
 * @param[in] obj The object.
 * @param[in] mode How window orientation should affect this widget.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_orientation_mode_set(Efl_Ui_Widget *obj, Efl_Ui_Widget_Orientation_Mode mode);

/**
 * @brief Whether the widget's automatic orientation is enabled or not.
 *
 * Orientation mode is used for widgets to change their style or send signals
 * based on the canvas rotation (i.e. the window orientation). If the
 * orientation mode is enabled, the widget will emit signals such as
 * "elm,state,orient,N" where @c N is one of 0, 90, 180, 270, depending on the
 * window orientation. Such signals may be handled by the theme in order to
 * provide a different look for the widget based on the canvas orientation.
 *
 * By default orientation mode is enabled.
 *
 * See also @ref elm_widget_on_orientation_update.
 *
 * @param[in] obj The object.
 *
 * @return How window orientation should affect this widget.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Efl_Ui_Widget_Orientation_Mode elm_widget_orientation_mode_get(const Efl_Ui_Widget *obj);

/**
 * @brief Virtual function handling canvas orientation changes.
 *
 * This method will be called recursively from the top widget (the window) to
 * all the children objects whenever the window rotation is changed. The given
 * @c rotation will be one of 0, 90, 180, 270 or the special value -1 if
 * @ref elm_widget_orientation_mode_get is @c disabled.
 *
 * If @ref elm_widget_orientation_mode_get is @c default, the widget
 * implementation will emit the signal "elm,state,orient,$R" will be emitted
 * (where @c R is the rotation angle in degrees).
 *
 * @note This function may be called even if the orientation has not actually
 * changed, like when a widget needs to be reconfigured.
 *
 * See also @ref elm_widget_orientation_mode_get.
 *
 * @param[in] obj The object.
 * @param[in] rotation Orientation in degrees: 0, 90, 180, 270 or -1 if
 * @ref elm_widget_orientation_mode_get is @c disabled.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_on_orientation_update(Efl_Ui_Widget *obj, int rotation);

/**
 * @brief Virtual function called when the widget becomes disabled.
 *
 * This may be triggered even if this widget is not disabled, as the parent
 * widget could be disabled and propagate its state.
 *
 * @param[in] obj The object.
 * @param[in] disabled The new value of @ref elm_widget_disabled_get.
 *
 * @return Indicates if the operation succeeded.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Eina_Bool elm_widget_on_disabled_update(Efl_Ui_Widget *obj, Eina_Bool disabled);

/**
 * @brief Virtual function called when the widget needs to re-apply its theme.
 *
 * This may be called when the object is first created, or whenever the widget
 * is modified in any way that may require a reload of the theme. This may
 * include but is not limited to scale, theme, or mirrored mode changes.
 *
 * @note even widgets not based on layouts may override this method to handle
 * widget updates (scale, mirrored mode, etc...).
 *
 * @param[in] obj The object.
 *
 * @return Indicates success, and if the current theme or default theme was
 * used.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Efl_Ui_Theme_Apply elm_widget_theme_apply(Efl_Ui_Widget *obj);

/**
 * @brief Hook function called when the @ref elm_widget_show_region_get is
 * changed.
 *
 * See also @ref elm_widget_show_region_get.
 *
 * @param[in] obj The object.
 * @param[in] func Region hook function
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_on_show_region_hook_set(Efl_Ui_Widget *obj, void *func_data, Efl_Ui_Scrollable_On_Show_Region func, Eina_Free_Cb func_free_cb);

/**
 * @brief Region inside the widget to show.
 *
 * See also @ref elm_widget_on_show_region_hook_set.
 *
 * Request parent scrollers to pan around so that this region of the widget
 * becomes visible.
 *
 * If @c force is @c true this will trigger scroller changes and the
 * @ref elm_widget_on_show_region_hook_set to be called even if the region is
 * unchanged.
 *
 * @param[in] obj The object.
 * @param[in] region The region to show.
 * @param[in] force Set to @c true to force show even if unchanged.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_show_region_set(Efl_Ui_Widget *obj, Eina_Rect region, Eina_Bool force);

/**
 * @brief Region inside the widget to show.
 *
 * See also @ref elm_widget_on_show_region_hook_set.
 *
 * Returns the current region to show.
 *
 * @param[in] obj The object.
 *
 * @return The region to show.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Eina_Rect elm_widget_show_region_get(const Efl_Ui_Widget *obj);

/** Push scroll hold
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_scroll_hold_push(Efl_Ui_Widget *obj);

/** Pop scroller hold
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_scroll_hold_pop(Efl_Ui_Widget *obj);

/** Push scroller freeze
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_scroll_freeze_push(Efl_Ui_Widget *obj);

/** Pop scroller freeze
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_scroll_freeze_pop(Efl_Ui_Widget *obj);

/**
 * @brief Region of interest inside this widget, that should be given priority
 * to be visible inside a scroller.
 *
 * When this widget or one of its subwidgets is given focus, this region should
 * be shown, which means any parent scroller should attempt to display the
 * given area of this widget. For instance, an entry given focus should scroll
 * to show the text cursor if that cursor moves. In this example, this region
 * defines the relative geometry of the cursor within the widget.
 *
 * @note The region is relative to the top-left corner of the widget, i.e. X,Y
 * start from 0,0 to indicate the top-left corner of the widget. W,H must be
 * greater or equal to 1 for this region to be taken into account, otherwise it
 * is ignored.
 *
 * See also @ref elm_widget_focus_region_show_mode_get
 *
 * @param[in] obj The object.
 *
 * @return The relative region to show. If width or height is <= 0 it will be
 * ignored, and no action will be taken.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Eina_Rect elm_widget_focus_region_get(const Efl_Ui_Widget *obj);

/**
 * @brief Defines how @ref elm_widget_show_region_get should be calculated.
 *
 * It is up to the implementation of @ref elm_widget_show_region_get to take
 * this value into account (or ignore it).
 *
 * @param[in] obj The object.
 * @param[in] mode Whether to show a specific item or the widget as a whole.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_focus_region_show_mode_set(Efl_Ui_Widget *obj, Efl_Ui_Interest_Region_Mode mode);

/**
 * @brief Defines how @ref elm_widget_show_region_get should be calculated.
 *
 * It is up to the implementation of @ref elm_widget_show_region_get to take
 * this value into account (or ignore it).
 *
 * @param[in] obj The object.
 *
 * @return Whether to show a specific item or the widget as a whole.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Efl_Ui_Interest_Region_Mode elm_widget_focus_region_show_mode_get(const Efl_Ui_Widget *obj);

/**
 * @brief The rectangle region to be highlighted on focus.
 *
 * This is a rectangle region where the focus highlight should be displayed.
 *
 * This is a read-only property.
 *
 * @param[in] obj The object.
 *
 * @return The rectangle area.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Eina_Rect elm_widget_focus_highlight_geometry_get(const Efl_Ui_Widget *obj);

/**
 * @brief Whether focus highlight is enabled or not.
 *
 * As of EFL 1.21 focus highlight properties apply to a single window, not a
 * single widget. As a consequence, calls to this function may be forwarded to
 * the parent window. Future versions of EFL may implement widget-specific
 * focus highlight properties.
 *
 * See also @ref elm_widget_top_get. See also
 * @ref elm_widget_focus_highlight_style_get. See also
 * @ref elm_widget_focus_highlight_animate_get.
 *
 * Set the enabled status for the focus highlight in a window.
 *
 * This function will enable or disable the focus highlight, regardless of the
 * global setting for it.
 *
 * @param[in] obj The object.
 * @param[in] enabled The enabled value for the highlight.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_focus_highlight_enabled_set(Efl_Ui_Widget *obj, Eina_Bool enabled);

/**
 * @brief Whether focus highlight is enabled or not.
 *
 * As of EFL 1.21 focus highlight properties apply to a single window, not a
 * single widget. As a consequence, calls to this function may be forwarded to
 * the parent window. Future versions of EFL may implement widget-specific
 * focus highlight properties.
 *
 * See also @ref elm_widget_top_get. See also
 * @ref elm_widget_focus_highlight_style_get. See also
 * @ref elm_widget_focus_highlight_animate_get.
 *
 * Get the enabled value of the focus highlight for this window.
 *
 * @param[in] obj The object.
 *
 * @return The enabled value for the highlight.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Eina_Bool elm_widget_focus_highlight_enabled_get(const Efl_Ui_Widget *obj);

/**
 * @brief Control the widget focus highlight style.
 *
 * If @c style is @c null, the default will be used.
 *
 * As of EFL 1.21 focus highlight properties apply to a single window, not a
 * single widget. As a consequence, calls to this function may be forwarded to
 * the parent window. Future versions of EFL may implement widget-specific
 * focus highlight properties.
 *
 * See also @ref elm_widget_top_get. See also
 * @ref elm_widget_focus_highlight_enabled_get. See also
 * @ref elm_widget_focus_highlight_animate_get.
 *
 * @param[in] obj The object.
 * @param[in] style The name of the focus highlight style.
 *
 * @return @c true on success, @c false otherwise.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Eina_Bool elm_widget_focus_highlight_style_set(Efl_Ui_Widget *obj, const char *style);

/**
 * @brief Control the widget focus highlight style.
 *
 * If @c style is @c null, the default will be used.
 *
 * As of EFL 1.21 focus highlight properties apply to a single window, not a
 * single widget. As a consequence, calls to this function may be forwarded to
 * the parent window. Future versions of EFL may implement widget-specific
 * focus highlight properties.
 *
 * See also @ref elm_widget_top_get. See also
 * @ref elm_widget_focus_highlight_enabled_get. See also
 * @ref elm_widget_focus_highlight_animate_get.
 *
 * @param[in] obj The object.
 *
 * @return The name of the focus highlight style.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI const char *elm_widget_focus_highlight_style_get(const Efl_Ui_Widget *obj);

/**
 * @brief Whether focus highlight should animate or not.
 *
 * As of EFL 1.21 focus highlight properties apply to a single window, not a
 * single widget. As a consequence, calls to this function may be forwarded to
 * the parent window. Future versions of EFL may implement widget-specific
 * focus highlight properties.
 *
 * See also @ref elm_widget_top_get. See also
 * @ref elm_widget_focus_highlight_style_get. See also
 * @ref elm_widget_focus_highlight_enabled_get.
 *
 * Set the animate status for the focus highlight for this window.
 *
 * This function will enable or disable the animation of focus highlight.
 *
 * @param[in] obj The object.
 * @param[in] animate The enabled value for the highlight animation.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI void elm_widget_focus_highlight_animate_set(Efl_Ui_Widget *obj, Eina_Bool animate);

/**
 * @brief Whether focus highlight should animate or not.
 *
 * As of EFL 1.21 focus highlight properties apply to a single window, not a
 * single widget. As a consequence, calls to this function may be forwarded to
 * the parent window. Future versions of EFL may implement widget-specific
 * focus highlight properties.
 *
 * See also @ref elm_widget_top_get. See also
 * @ref elm_widget_focus_highlight_style_get. See also
 * @ref elm_widget_focus_highlight_enabled_get.
 *
 * Get the animate value of the focus highlight for this window.
 *
 * @param[in] obj The object.
 *
 * @return The enabled value for the highlight animation.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Eina_Bool elm_widget_focus_highlight_animate_get(const Efl_Ui_Widget *obj);

/**
 * @brief Register focus with the given configuration.
 *
 * The implementation can feel free to change the logical flag as it wants, but
 * other than that it should strictly keep the configuration.
 *
 * The implementation in elm.widget updates the current state into what is
 * passed as configured state, respecting manager changes, registeration and
 * unregistration based on if it should be registered or unregistered.
 *
 * A manager field that is @c null means that the widget should not or was not
 * registered.
 *
 * @param[in] obj The object.
 * @param[in] current_state The focus manager to register with.
 * @param[in,out] configured_state The evalulated Focus state that should be
 * used.
 * @param[in] redirect A redirect that will be set by the elm.widget
 * implementation.
 *
 * @return Returns whether the widget is registered or not.
 *
 * @ingroup Efl_Ui_Widget
 */
EAPI Eina_Bool elm_widget_focus_state_apply(Efl_Ui_Widget *obj, Efl_Ui_Widget_Focus_State current_state, Efl_Ui_Widget_Focus_State *configured_state, Efl_Ui_Widget *redirect);

#endif
