package ro.sync.lexer.css;
import ro.sync.lexer.AbstractLexer;

@SuppressWarnings("unused")
%%

%public 
%class CSSLexer
%extends AbstractLexer
%unicode
%char
%type ro.sync.lexer.Symbol
 
%ignorecase
%scanerror ro.sync.lexer.LexerException

%{
    private static final byte SYM_TEXT               = CSSTokens.TEXT;
    private static final byte SYM_INVALID            = CSSTokens.INVALID;
    private static final byte SYM_SELECTOR           = CSSTokens.SELECTOR;
    private static final byte SYM_SELECTOR_CLASS     = CSSTokens.SELECTOR_CLASS;
    private static final byte SYM_SELECTOR_ID        = CSSTokens.SELECTOR_ID;
    
    private static final byte SYM_CURLY_BRACKET      = CSSTokens.CURLY_BRACKET;
    private static final byte SYM_SQUARE_BRACKET     = CSSTokens.SQUARE_BRACKET;
    private static final byte SYM_BRACKET            = CSSTokens.BRACKET;
    private static final byte SYM_EQUAL              = CSSTokens.OPERATOR;
    private static final byte SYM_COLON              = CSSTokens.OPERATOR;
    private static final byte SYM_DOUBLECOLON        = CSSTokens.OPERATOR;
    private static final byte SYM_SEMICOLON          = CSSTokens.OPERATOR;
    private static final byte SYM_AT                 = CSSTokens.OPERATOR;
    
    private static final byte SYM_LIST_SEPARATOR     = CSSTokens.LIST_SEPARATOR;
    
    private static final byte SYM_PROPERTY           = CSSTokens.PROPERTY;
    private static final byte SYM_VALUE              = CSSTokens.VALUE;
    private static final byte SYM_IMPORTANT          = CSSTokens.IMPORTANT;
    private static final byte SYM_STRING_DQ          = CSSTokens.STRING_DQ;
    private static final byte SYM_STRING_SQ          = CSSTokens.STRING_SQ;
    
    private static final byte SYM_COMMENT_START      = CSSTokens.COMMENT;
    private static final byte SYM_COMMENT            = CSSTokens.COMMENT;
    private static final byte SYM_COMMENT_END        = CSSTokens.COMMENT;

    private static final byte SYM_COMMENT_XML_START  = CSSTokens.COMMENT_XML;
    private static final byte SYM_COMMENT_XML_END    = CSSTokens.COMMENT_XML;

    private static final byte SYM_CDATA_XML_START  = CSSTokens.CDATA_XML;
    private static final byte SYM_CDATA_XML_END    = CSSTokens.CDATA_XML;

    private static final byte SYM_CHARSET            = CSSTokens.AT_RULE;
    private static final byte SYM_FONT_FACE          = CSSTokens.AT_RULE;

    private static final byte SYM_MEDIA              = CSSTokens.AT_RULE;
    private static final byte SYM_MEDIA_LIST         = CSSTokens.AT_RULE_CONTENT;

    private static final byte SYM_IMPORT             = CSSTokens.AT_RULE;
    private static final byte SYM_IMPORT_CONTENT     = CSSTokens.AT_RULE_CONTENT;

    private static final byte SYM_PAGE               = CSSTokens.AT_RULE;
    private static final byte SYM_PAGE_NAME          = CSSTokens.AT_RULE_CONTENT;

    private static final byte SYM_NAMESPACE          = CSSTokens.AT_RULE;
    private static final byte SYM_NAMESPACE_CONTENT  = CSSTokens.AT_RULE_CONTENT;
    private static final byte SYM_NAMESPACE_SEPARATOR= CSSTokens.OPERATOR;

    private static final byte SYM_TILDA              = CSSTokens.OPERATOR;
    private static final byte SYM_SHARP              = CSSTokens.OPERATOR;
    private static final byte SYM_DOT                = CSSTokens.OPERATOR;
    private static final byte SYM_GT                 = CSSTokens.OPERATOR;
    private static final byte SYM_PLUS               = CSSTokens.OPERATOR;
    private static final byte SYM_INCLUDES           = CSSTokens.OPERATOR;
    private static final byte SYM_DASHMATCH          = CSSTokens.OPERATOR;
    private static final byte SYM_STARTS_WITH        = CSSTokens.OPERATOR;
    private static final byte SYM_ENDS_WITH          = CSSTokens.OPERATOR;
    private static final byte SYM_CONTAINS           = CSSTokens.OPERATOR;
    /**
     * The alternate state where we should return after unclosed strings.
     */
    private int alternateStateForUnclosedStrings = 0;
    
    private String lastYYText = "";
    
    /**
     * Create an empty lexer, yyreset will be called later to reset and assign
     * the reader
     */
    public CSSLexer() {
        super();
    }
    
    public String getName() {
      return CSS_LEXER;
    }
%}

%xstate COMMENT, SELECTOR_BODY, DECLARATION, VALUE, DQ_STRING, SQ_STRING 
%xstate MEDIA_LIST, IMPORT, PAGE, NAMESPACE

/* Any character. Anything interesting must be handled above.*/
Char = .
GeneralChar = [^</,{\->+~|\'\"\[\]\@ \t:=$*\.#\(\)]

DQStringContentPart =  ([^\"])
SQStringContentPart =  ([^\'])

%%

<YYINITIAL> {
    "@media"                     {
                                     cLen = 0;
                                     yybegin(MEDIA_LIST);
                                     return symbol(SYM_MEDIA);                                    
                                 }
    "@charset"                   {   return symbol(SYM_CHARSET);        }
    "@font-face"                 {   return symbol(SYM_FONT_FACE);      }
    "@import"                    {
                                     cLen = 0;
                                     // Take care of imports.
                                     yybegin(IMPORT);                                     
                                     return symbol(SYM_IMPORT);
                                 }
    "@namespace"                 {
                                     cLen = 0;
                                     // Take care of imports.
                                     yybegin(NAMESPACE);                                     
                                     return symbol(SYM_NAMESPACE);
                                 }
    "@page"                      {
                                     cLen = 0;
                                     // Take care of pages.
                                     yybegin(PAGE);                                
                                     return symbol(SYM_PAGE);
                                 }
    "/*"                         {
                                     cLen = 0;
                                     // Save state to return to.
                                     pushState(YYINITIAL);
                                     yybegin(COMMENT);
                                     return symbol(SYM_COMMENT_START);
                                 }
    "<!--"                       {   return symbol(SYM_COMMENT_XML_START);  }
    "-->"                        {   return symbol(SYM_COMMENT_XML_END);    }
    "<![CDATA["                  {   return symbol(SYM_CDATA_XML_START);    }
    "]]>"                        {   return symbol(SYM_CDATA_XML_END);      }
    "{"                          {
                                     // Reset the char counter
                                     cLen = 0;
                                     yybegin(DECLARATION);
                                     return symbol(SYM_CURLY_BRACKET);
                                 }
    "}"                          {
                                     // Hack. This is usually consummed in the DECLARATION state.
                                     // However, when using @media, we return in YYINTITIAL after the
                                     // media block starts, so we can read the ending "}" in the 
                                     // YYINITIAL state.
                                     return symbol(SYM_CURLY_BRACKET);
                                 }                                                                  
    // Selector separator
    ","                          {   return symbol(SYM_LIST_SEPARATOR); }    
    // Namespace separator
    "|"                          {   return symbol(SYM_NAMESPACE_SEPARATOR); }    
    "="                          {   return symbol(SYM_EQUAL);          }
    ">"                          {   return symbol(SYM_GT);             }
    "~"                          {   return symbol(SYM_TILDA);          }
    "-"                          {   return symbol(SYM_TEXT);           }
    "+"                          {   return symbol(SYM_PLUS);           }
    "@"                          {   return symbol(SYM_AT);             }
    ";"                          {   return symbol(SYM_SEMICOLON);      }
    ":"                          {   return symbol(SYM_COLON);          }
    "~="                         {   return symbol(SYM_INCLUDES);       }
    "|="                         {   return symbol(SYM_DASHMATCH);      }
    "^="                         {   return symbol(SYM_STARTS_WITH);    }
    "$="                         {   return symbol(SYM_ENDS_WITH);      }
    "*="                         {   return symbol(SYM_CONTAINS);       }
    "::"                         {   return symbol(SYM_DOUBLECOLON);    }
    "\""                         {
                                     cLen = 1;
                                     pushState(YYINITIAL);
                                     yybegin(DQ_STRING);
                                 }
    "\'"                         {
                                     cLen = 1;
                                     pushState(YYINITIAL);
                                     yybegin(SQ_STRING);
                                 }
    "["|"]"                      {   return symbol(SYM_SQUARE_BRACKET); }
    "("|")"                      {   return symbol(SYM_BRACKET);        }
    "*"                          {   return symbol(SYM_TEXT);           }
    // White spaces are emitted separatelly.
    [ \t]+                       {   return symbol(SYM_TEXT);           }
	// This is Text
	// Match anything else different from the markup.
    "." {GeneralChar}*           {   return symbol(SYM_SELECTOR_CLASS); }
    "#" {GeneralChar}*           {   return symbol(SYM_SELECTOR_ID);    }
    {GeneralChar}*               {   return symbol(SYM_SELECTOR);       }
}

<DECLARATION> {
  ":"                            {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(1);
                                        return flush(SYM_PROPERTY);
                                    } else {
                                        yybegin(VALUE);
                                        cLen = 0;
                                        return symbol(SYM_COLON);                                     
                                    }
                                 }
   "/*"                          {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(2);
                                        return flush(SYM_PROPERTY);
                                    } else {
                                        // Save state to return to.
                                        pushState(DECLARATION);
                                        yybegin(COMMENT);
                                        // Reset the char counter
                                        cLen = 0;
                                        return symbol(SYM_COMMENT_START);
                                    }                               
                                     
                                 }
  "}"                            {
                                    // Normal ending of a declaration.
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(1);
                                        return flush(SYM_PROPERTY);
                                    } else {
                                        // No buffer
                                        // Change state
                                        yybegin(YYINITIAL);
                                        return symbol(SYM_CURLY_BRACKET);
                                    }                               
                                 }                                 
  {Char}                         {   cLen ++;   }                                   
  <<EOF>>                        {   return flush(SYM_PROPERTY);    }
}

<VALUE> {
   ";"                           {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(1);
                                        return flush(SYM_VALUE);
                                    } else {
                                        yybegin(DECLARATION);
	                                    cLen = 0;
	                                    return symbol(SYM_SEMICOLON);                            
                                    }                               
                                 }  
  "}"                            {
                                    // Normal ending of a declaration. The value is not separated 
                                    // from the bracket by a ";". This can be valid.
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(1);
                                        return flush(SYM_VALUE);
                                    } else {
                                        // No buffer
                                        // Change state
                                        yybegin(YYINITIAL);
                                        return symbol(SYM_CURLY_BRACKET);
                                    }                               
                                 }
    "\""                         {
                                     if (cLen > 0) {
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(1);
                                        return flush(SYM_VALUE);
                                     } else {
	                                     cLen = 1;
	                                     pushState(VALUE);
                                         alternateStateForUnclosedStrings = DECLARATION;
	                                     yybegin(DQ_STRING);
                                     }   
                                 }
    "\'"                         {
                                     if (cLen > 0) {
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(1);
                                        return flush(SYM_VALUE);
                                     } else {
                                         cLen = 1;
                                         pushState(VALUE);
                                         alternateStateForUnclosedStrings = DECLARATION;
                                         yybegin(SQ_STRING);
                                     }
                                 }
   "/*"                          {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(2);
                                        return flush(SYM_VALUE);
                                    } else {
                                        // Save state to return to.
                                        pushState(VALUE);
                                        yybegin(COMMENT);
                                        // Reset the char counter
                                        cLen = 0;
                                        return symbol(SYM_COMMENT_START);
                                    }                               
                                 }
   "!"                           {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(1);
                                        return flush(SYM_VALUE);
                                    } else {
                                        return symbol(SYM_IMPORTANT);
                                    }                               
                                 }
   "important"                   {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(yylength());
                                        return flush(SYM_VALUE);
                                    } else {
                                        return symbol(SYM_IMPORTANT);
                                    }                               
                                     
                                 }
                                 
  {Char}                         {   cLen ++;   }                                   
  <<EOF>>                        {   return flush(SYM_VALUE);    }
}

<DQ_STRING> {
    "\""                         {
                                     if ("\\".equals(lastYYText)) {
                                         cLen++;
                                         lastYYText = yytext();
                                     } else {
                                         yybegin(popState());
                                         cLen++;
                                         lastYYText = yytext();
                                         return flush(SYM_STRING_DQ);
                                     }
                                 }
    "\\"                         {
                                     cLen++;
                                     if ("\\".equals(lastYYText)) {
                                        lastYYText = "";
                                     } else {
                                        lastYYText = yytext();
                                     }
                                 }
    [^\"\\]                      {   
                                     cLen++;
                                     lastYYText = yytext();
                                 }
    <<EOF>>                      {
                                     try {
	                                     if ("\\".equals(lastYYText)) {
	                                         // Keep the state, but flush the string content.
	                                         return flush(SYM_STRING_DQ);
	                                     } else {
	                                         if (cLen > 0) {
			                                     // Remove the state since it s unnecessary now...
			                                     popState();
			                                     // Switch to the alternate state after unclosed string.
			                                     yybegin(alternateStateForUnclosedStrings);
		                                     }
		                                     return flush(SYM_STRING_DQ);
	                                     }
                                     } finally {
                                         lastYYText = null;
                                     }
                                 }
}

<SQ_STRING> {
    "'"                          {
                                     if ("\\".equals(lastYYText)) {
                                         cLen++;
                                         lastYYText = yytext();
                                     } else {
	                                     yybegin(popState());
	                                     cLen++;
	                                     return flush(SYM_STRING_SQ);
                                     }
                                 }
    "\\"                         {
                                     cLen++;
                                     if ("\\".equals(lastYYText)) {
                                        lastYYText = "";
                                     } else {
                                        lastYYText = yytext();
                                     }
                                 }
    [^\'\\]                      {   
                                     cLen++;
                                     lastYYText = yytext();
                                 }
    <<EOF>>                      {
                                     try {
                                         if ("\\".equals(lastYYText)) {
                                             // Keep the state, but flush the string content.
                                             return flush(SYM_STRING_SQ);
                                         } else {
                                             if (cLen > 0) {
                                                 // Remove the state since it s unnecessary now...
                                                 popState();
                                                 // Switch to the alternate state after unclosed string.
                                                 yybegin(alternateStateForUnclosedStrings);
                                             }
                                             return flush(SYM_STRING_SQ);
                                         }
                                     } finally {
                                         lastYYText = null;
                                     }
                                  }
}

<COMMENT> {
  "*/"                           {
                                     yybegin(popState());
                                     return symbol(SYM_COMMENT_END);                                     
                                 }
  {Char}                         {   cLen ++;   }                                 
  ~"*/"                          {
                                     yypushback(2);
                                     return symbol(SYM_COMMENT);
                                 }
  <<EOF>>                        {   return flush(SYM_COMMENT);    }
}

<MEDIA_LIST>  {
    ","                          {                                     
                                     if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(1);
                                        return flush(SYM_MEDIA_LIST);
                                    } else {
                                        return symbol(SYM_LIST_SEPARATOR);
                                    }                                         
                                 }
    "{"                          {
                                    // We switch to YYINIITAL, it is very similar to the media content.
                                    // In YYINITIAL, we will discard extra "}"
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(1);
                                        return flush(SYM_MEDIA_LIST);
                                    } else {
                                        yybegin(YYINITIAL);
                                        return symbol(SYM_CURLY_BRACKET);                                     
                                    }                 
                                 }
    "/*"                         {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(2);
                                        return flush(SYM_MEDIA_LIST);
                                    } else {
                                        // Save state to return to.
                                        pushState(MEDIA_LIST);
                                        yybegin(COMMENT);
                                        // Reset the char counter
                                        cLen = 0;
                                        return symbol(SYM_COMMENT_START);
                                    }                               
                                 }
    {Char}                       {   cLen ++;  }
    <<EOF>>                      {
                                    return flush(SYM_MEDIA_LIST);
                                 }                                               
}

<PAGE>  {
    ":"                          {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(1);
                                        return flush(SYM_PAGE_NAME);
                                    } else {
                                        cLen = 0;
                                        return symbol(SYM_COLON);                            
                                    }      
                                 }
    "{"                          {
                                    // We switch to YYINIITAL, it is very similar to the page content.
                                    // In YYINITIAL, we will discard extra "}"
                                    yypushback(1);
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        return flush(SYM_PAGE_NAME);
                                    } else {
                                        yybegin(YYINITIAL);
                                    }                 
                                 }
    "/*"                         {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(2);
                                        return flush(SYM_PAGE_NAME);
                                    } else {
                                        // Save state to return to.
                                        pushState(PAGE);
                                        yybegin(COMMENT);
                                        // Reset the char counter
                                        cLen = 0;
                                        return symbol(SYM_COMMENT_START);
                                    }                               
                                 }
    {Char}                       {   cLen ++;  }
    <<EOF>>                      {
                                    return flush(SYM_PAGE_NAME);
                                 }                                               
}

<NAMESPACE>  {
    ";"                          {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(1);
                                        return flush(SYM_NAMESPACE_CONTENT);
                                    } else {
                                        cLen = 0;
                                        yybegin(YYINITIAL);
                                        return symbol(SYM_COLON);
                                    }      
                                 }
    "(" | ")"                    {
                                    if(cLen > 0){
                                        yypushback(1);
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        return flush(SYM_NAMESPACE_CONTENT);
                                    } else {
                                        return symbol(SYM_NAMESPACE_CONTENT);
                                    }                 
                                 }
    "/*"                         {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(2);
                                        return flush(SYM_NAMESPACE_CONTENT);
                                    } else {
                                        // Save state to return to.
                                        pushState(NAMESPACE);
                                        yybegin(COMMENT);
                                        // Reset the char counter
                                        cLen = 0;
                                        return symbol(SYM_COMMENT_START);
                                    }                               
                                 }
    
    "\""                         {
                                    if (cLen > 0) {
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(yylength());
                                        return flush(SYM_NAMESPACE_CONTENT);
                                    } else {
                                        cLen = 1;
                                        pushState(NAMESPACE);
                                        alternateStateForUnclosedStrings = YYINITIAL;
                                        yybegin(DQ_STRING);
                                    }
                                 }
    "\'"                         {
                                    if (cLen > 0) {
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(yylength());
                                        return flush(SYM_NAMESPACE_CONTENT);
                                    } else {
                                        cLen = 1;
                                        pushState(NAMESPACE);
                                        alternateStateForUnclosedStrings = YYINITIAL;
                                        yybegin(SQ_STRING);
                                    }
                                 }
    {DQStringContentPart}
    | {SQStringContentPart}      {   cLen ++;  }
    
    <<EOF>>                      {
                                    yybegin(YYINITIAL);
                                    return flush(SYM_NAMESPACE_CONTENT);
                                 }                                               
}

<IMPORT>                    {
   ";"                           {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(1);
                                        return flush(SYM_IMPORT_CONTENT);
                                    } else {
                                        yybegin(YYINITIAL);
                                        cLen = 0;
                                        return symbol(SYM_SEMICOLON);                            
                                    }                               
                                 }  
  "/*"                           {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        yypushback(2);
                                        return flush(SYM_IMPORT_CONTENT);
                                    } else {
                                        // Save state to return to.
                                        pushState(IMPORT);
                                        yybegin(COMMENT);
                                        // Reset the char counter
                                        cLen = 0;
                                        return symbol(SYM_COMMENT_START);
                                    }                               
                                 }
    "\""                         {
                                    if (cLen > 0) {
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(yylength());
                                        return flush(SYM_IMPORT_CONTENT);
                                    } else {
                                        cLen = 1;
                                        pushState(NAMESPACE);
                                        alternateStateForUnclosedStrings = YYINITIAL;
                                        yybegin(DQ_STRING);
                                    }
                                 }
    "\'"                         {
                                    if (cLen > 0) {
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the bracket in the same state.
                                        yypushback(yylength());
                                        return flush(SYM_IMPORT_CONTENT);
                                    } else {
                                        cLen = 1;
                                        pushState(NAMESPACE);
                                        alternateStateForUnclosedStrings = YYINITIAL;
                                        yybegin(SQ_STRING);
                                    }
                                 }
  {Char}                         {   cLen ++;   }                                   
  <<EOF>>                        {   
                                     yybegin(YYINITIAL);
                                     return flush(SYM_IMPORT_CONTENT);
                                 }
}