package ro.sync.lexer.xqueryxpath;

import ro.sync.lexer.AbstractLexer;
import ro.sync.lexer.LexicalState;
import ro.sync.lexer.LexicalStateUnlimittedStack;

// http://www.w3.org/TR/xquery-xpath-parsing/
// http://www.w3.org/TR/xquery-30/
// http://www.w3.org/TR/xpath-30/
@SuppressWarnings("unused")
%%

%public 
%class XQueryXPathLexer
%extends AbstractLexer
%unicode
%char
%type ro.sync.lexer.Symbol

%caseless
%scanerror ro.sync.lexer.LexerException

%{
    private static final byte SYM_TEXT              = XQueryXPathTokens.TEXT;
    private static final byte SYM_KEYWORD           = XQueryXPathTokens.KEYWORD;
    private static final byte SYM_AXIS              = XQueryXPathTokens.AXIS;
    private static final byte SYM_IDENTIFIER        = XQueryXPathTokens.IDENTIFIER;
    private static final byte SYM_FUNCTION          = XQueryXPathTokens.FUNCTION;
    private static final byte SYM_VARIABLE          = XQueryXPathTokens.VARIABLE;
    private static final byte SYM_ATTRIBUTE_SELECTOR= XQueryXPathTokens.ATTRIBUTE_SELECTOR;
    private static final byte SYM_ANNOTATION        = XQueryXPathTokens.ANNOTATION;    
    
    private static final byte SYM_OPERATOR          = XQueryXPathTokens.OPERATOR;
    private static final byte SYM_COMMENT           = XQueryXPathTokens.COMMENT;
    private static final byte SYM_PI                = XQueryXPathTokens.PI;
    private static final byte SYM_NUMBER            = XQueryXPathTokens.NUMBER;
    private static final byte SYM_DELIMITER         = XQueryXPathTokens.DELIMITER;
    private static final byte SYM_PRAGMA            = XQueryXPathTokens.PRAGMA;
    private static final byte SYM_PRAGMA_CONTENT    = XQueryXPathTokens.PRAGMA;
    private static final byte SYM_COMMA             = XQueryXPathTokens.COMMA;
    private static final byte SYM_DOT               = XQueryXPathTokens.STAR;
    private static final byte SYM_DOTDOT            = XQueryXPathTokens.STAR;
    private static final byte SYM_STAR              = XQueryXPathTokens.STAR;
    private static final byte SYM_ASK               = XQueryXPathTokens.STAR;
    private static final byte SYM_PLUS              = XQueryXPathTokens.STAR;
    private static final byte SYM_SEMICOLON         = XQueryXPathTokens.SEMICOLON;
    private static final byte SYM_BRACKET           = XQueryXPathTokens.BRACKET;
    private static final byte SYM_SQUARE_BRACKET    = XQueryXPathTokens.SQUARE_BRACKET;
    private static final byte SYM_CURLY_BRACKET     = XQueryXPathTokens.CURLY_BRACKET;
    private static final byte SYM_STRING_DQ         = XQueryXPathTokens.STRING_DQ;
    private static final byte SYM_STRING_SQ         = XQueryXPathTokens.STRING_SQ;
    
    private static final byte SYM_XML_COMMENT       = XQueryXPathTokens.XML_COMMENT;
    private static final byte SYM_CDATA             = XQueryXPathTokens.CDATA;
    private static final byte SYM_ENTITY            = XQueryXPathTokens.OPERATOR;
    
    private static final byte SYM_EQUAL             = XQueryXPathTokens.EQUAL;
    
    private static final byte SYM_XML_TAG           = XQueryXPathTokens.XML_TAG;     
    private static final byte SYM_XML_ELEMENT       = XQueryXPathTokens.XML_TAG;
    private static final byte SYM_XML_ATTRIBUTE     = XQueryXPathTokens.XML_ATTRIBUTE;
    private static final byte SYM_ATTRIBUTE_VALUE   = XQueryXPathTokens.ATTRIBUTE_VALUE;
    
    /**
     * Create an empty lexer, yyreset will be called later to reset and assign
     * the reader
     */
    public XQueryXPathLexer() {
        super();
    }
    
    public String getName() {
      return XQUERY_XPATH_LEXER;
    }
    
    /**
     * Create a new lexical state to be used during tokenization.
     * It is supposed to be equivalent with YYINITIAL...
     * 
     * @return  The lexical state.
     */
    protected LexicalState createLexicalState() {
      return new LexicalStateUnlimittedStack();
    }
    
    private static final LexicalState INITIAL_STATE = new LexicalStateUnlimittedStack();
    
    /**
     * Create a new lexical state to be used during tokenization.
     * It is supposed to be equivalent with YYINITIAL...
     * 
     * @return  The lexical state.
     */
    protected LexicalState getInitialLexicalState() {
      return INITIAL_STATE;
    }
    
%}

%xstate COMMENT, OPERATOR, QNAME, DQ_STRING_MULTILINE, SQ_STRING_MULTILINE, SINGLE_TYPE, ITEM_TYPE
%xstate VARNAME, KINDTEST, OCCURENCE_INDICATOR, PRAGMA, PRAGMA_CONTENTS, NAMESPACE_DECL
%xstate NAMESPACE_END_EXPECTED
%xstate NAMESPACE_KEYWORD, XML_COMMENT, PROCESSING_INSTRUCTION, CDATA, START_TAG, XML_ATTRIBUTES
%xstate QUOT_ATTRIBUTE_CONTENT, APOS_ATTRIBUTE_CONTENT, ELEMENT_CONTENT, END_TAG, XMLSPACE_DECL
%xstate XQUERY_VERSION, OPTION, KNOWN_FUNC, UNKNOWN_FUNC, FAKE_FUNC, ITEM, VOID, CATCH_ML, CATCH
%xstate ANNOTATION, WINDOW_CLAUSE


///////////////////////////////////////// Keywords ////////////////////////////////////////////////

KW_Allowing = "Allowing"
KW_Ancestor = "ancestor"
KW_Ancestor_or_self = "ancestor-or-self"
KW_And = "and"
KW_As = "as"
KW_Ascending = "ascending"
KW_At = "at"
KW_Attribute = "attribute"
KW_Base_uri = "base-uri"
KW_Boundary_space = "boundary-space"
KW_By = "by"
KW_Case = "case"
KW_Cast = "cast"
KW_Castable = "castable"
KW_Catch = "catch"
KW_Child = "child"
KW_Collation = "collation"
KW_Comment = "comment"
KW_Construction = "construction"
KW_Copy_namespaces = "copy-namespaces"
KW_Count = "count"
KW_Declare = "declare"
KW_Define = "define"
KW_Default = "default"
KW_Descendant = "descendant"
KW_Descendant_or_self = "descendant-or-self"
KW_Descending = "descending"
KW_Div = "div"
KW_Document = "document"
KW_Document_node = "document-node"
KW_Element = "element"
KW_Else = "else"
KW_Empty = "empty"
KW_Encoding = "encoding"
KW_End = "end"
KW_Eq = "eq"
KW_Every = "every"
KW_Except = "except"
KW_External = "external"
KW_Following = "following"
KW_Following_sibling = "following-sibling"
KW_For = "for"
KW_Function = "function"
KW_Ge = "ge"
KW_Greatest = "greatest"
KW_Group = "group"
KW_Gt = "gt"
KW_Idiv = "idiv"
KW_If = "if"
KW_Import = "import"
KW_In = "in"
KW_Inherit = "inherit"
KW_Instance = "instance"
KW_Intersect = "intersect"
KW_Is = "is"
KW_Item = "item"
KW_Lax = "lax"
KW_Le = "le"
KW_Least = "least"
KW_Let = "let"
KW_Lt = "lt"
KW_Mod = "mod"
KW_Module = "module"
KW_Namespace = "namespace"
KW_Ne = "ne"
KW_Next = "next"
KW_No_inherit = "no-inherit"
KW_No_preserve = "no-preserve"
KW_Node = "node"
KW_Of = "of"
KW_Only = "only"
KW_Option = "option"
KW_Or = "or"
KW_Order = "order"
KW_Ordered = "ordered"
KW_Ordering = "ordering"
KW_Parent = "parent"
KW_Preceding = "preceding"
KW_Preceding_sibling = "preceding-sibling"
KW_Preserve = "preserve"
KW_Previous = "previous"
KW_Private = "private"
KW_Processing_instruction = "processing-instruction"
KW_Return = "return"
KW_Satisfies = "satisfies"
KW_Schema = "schema"
KW_Schema_attribute = "schema-attribute"
KW_Schema_element = "schema-element"
KW_Self = "self"
KW_Sequential = "sequential"
KW_Sliding = "sliding"
KW_Some = "some"
KW_Stable = "stable"
KW_Start = "start"
KW_Strict = "strict"
KW_Strip = "strip"
KW_Switch = "switch"
KW_Text = "text"
KW_Then = "then"
KW_To = "to"
KW_Treat = "treat"
KW_Try = "try"
KW_Tumbling = "tumbling"
KW_Typeswitch = "typeswitch"
KW_Union = "union"
KW_Unordered = "unordered"
KW_Validate = "validate"
KW_Variable = "variable"
KW_Version = "version"
KW_Void = "void"
KW_When = "when"
KW_Where = "where"
KW_Window = "window"
KW_Xquery = "xquery"

//////////////////////////////////////// Axis keyword //////////////////////////////////////////////

AxisKeyword = {KW_Ancestor_or_self} | {KW_Ancestor} | {KW_Attribute} | {KW_Child}
            | {KW_Descendant_or_self} | {KW_Descendant} | {KW_Following_sibling} | {KW_Following}
            | {KW_Namespace} | {KW_Parent} | {KW_Preceding_sibling} | {KW_Preceding} | {KW_Self}

Axis = {AxisKeyword} "::"

//////////////////////////////////////// Known function names //////////////////////////////////////

Known_function_name = {KW_Element}
    | {KW_Attribute}
    | {KW_Schema_element}
    | {KW_Schema_attribute}
    | {KW_Comment}
    | {KW_Text} 
    | {KW_Processing_instruction}
    | {KW_Node}
    | {KW_Document_node}

//////////////////////////////////////////// NCName ////////////////////////////////////////////////

NCNameStartChar = [A-Za-z_] | [\u00C0-\u00D6] | [\u00D8-\u00F6] | [\u00F8-\u02FF] | [\u0370-\u037D]
                | [\u037F-\u1FFF] | [\u200C-\u200D] | [\u2070-\u218F] | [\u2C00-\u2FEF]
                | [\u3001-\uD7FF] | [\uF900-\uFDCF] | [\uFDF0-\uFFFD]


NCNameChar = {NCNameStartChar} | [-\.0-9] | \u00B7 | [\u0300-\u036F] | [\u203F-\u2040]

NCNonNameChar = [^A-Za-z_\-+,;:=!?*/\"\'<>\(\[\{@$%&\u00C0-\u00D6\u00D8-\u00F6\u00F8-\u02FF\u0370-\u037D\u037F-\u1FFF\u200C-\u200D\u2070-\u218F\u2C00-\u2FEF\u3001-\uD7FF\uF900-\uFDCF\uFDF0-\uFFFD\.0-9\u00B7\u0300-\u036F\u203F-\u2040]

NCName = {NCNameStartChar} {NCNameChar}*
QName = {NCName} | ({NCName} ":" {NCName}) | ("*:" {NCName}) | ({NCName} ":*")
Entity = "&" {NCName} ";" 
// Example:  3 &lt;= 5
EntityOperator = "&" {NCName} ";=" 

///////////////////////////////////////////// Strings //////////////////////////////////////////////

DQStringContent =  ([^\"] | \"\")*
DQUnclosedString =  \" {DQStringContent}
DQString =  {DQUnclosedString} \"
SQStringContent =  ([^\'] | \'\')*
SQUnclosedString =  \' {SQStringContent}
SQString =  {SQUnclosedString} \'
String = ({DQString} | {SQString})

//////////////////////////////////////////// Numbers ///////////////////////////////////////////////

Digit = [0-9]

// Integers
Integer = {Digit}+

// Decimals
Decimal = (\. {Digit}+) | ({Digit}+ \. {Digit}*)

// Doubles
Exponent = [eE] [+-]? {Digit}+       
Double = ((\. {Digit}+) | ({Digit}+ ("." {Digit}*)?)) {Exponent}

// All numbers
Number = {Integer} | {Decimal} | {Double}

//////////////////////////////////////// Annotation ////////////////////////////////////////////////

AnnotationParam = {String} | {Integer}
Annotation = ("%" {QName} ("(" {AnnotationParam} ("," {AnnotationParam})* ")")?)

//////////////////////////////////////// Whitespaces ///////////////////////////////////////////////

WS = [ \t]

//////////////////////////////////////// All characters ////////////////////////////////////////////

Char = .

%% 

<YYINITIAL> 
{
////////////////////////////////////////////////////////////

// Switching to Operator
    {Number}                    {
                                    yybegin(OPERATOR);   
                                    return symbol(SYM_NUMBER);
                                }
    ".."                        {
                                    yybegin(OPERATOR);
                                    return symbol(SYM_DOTDOT);
                                }
    "."                         {
                                    yybegin(OPERATOR);
                                    return symbol(SYM_DOT);
                                }
    "*"                         {
                                    yybegin(OPERATOR);
                                    return symbol(SYM_STAR);
                                }
    ")"                         {
                                    yybegin(OPERATOR);   
                                    return symbol(SYM_BRACKET);
                                }
          
    ({KW_Declare} {WS}+ {KW_Construction} {NCNonNameChar}*)
    | ({KW_Declare} {WS}+ {KW_Default} {WS}+ {KW_Order} {NCNonNameChar}*)
                                {
                                    
                                    yybegin(OPERATOR);
                                    return symbol(SYM_KEYWORD);
                                }
    
    {DQString}                  {   
                                    yybegin(OPERATOR);
                                    return symbol(SYM_STRING_DQ);
                                }
    {DQUnclosedString}          {   
                                    // At the end of the string state will switch to OPERATOR
                                    pushState(OPERATOR);
                                    yybegin(DQ_STRING_MULTILINE);   
                                    return symbol(SYM_STRING_DQ);       
                                }
    
    {SQString}                  {   
                                    yybegin(OPERATOR);
                                    return symbol(SYM_STRING_SQ);       
                                }
    
    {SQUnclosedString}          {   
                                    // At the end of the string state will switch to OPERATOR
                                    pushState(OPERATOR);
                                    yybegin(SQ_STRING_MULTILINE);
                                    return symbol(SYM_STRING_SQ);       
                                }
                                
    {KW_Try}                    {   
                                    popState();
                                    return symbol(SYM_KEYWORD);
                                }
                                
    {KW_Catch} {WS}* "("        {
                                    yypushback(yylength());
                                    pushState(OPERATOR);
                                    yybegin(CATCH_ML);
                                }
                                
    {KW_Catch}                  {
                                    yybegin(CATCH);
                                    return symbol(SYM_KEYWORD);
                                }                                
                                
    ("*:" {NCName})
    | ({NCName} ":*")
    | {QName}                   {
                                    yybegin(OPERATOR);
                                    return symbol(SYM_IDENTIFIER);
                                }
    // Switching to NAMESPACE_DECL
    (({KW_Declare} {WS}+ {KW_Default} {WS}+ {KW_Collation})
    | ({KW_Default} {WS}+ {KW_Collation})
    | ({KW_Declare} {WS}+ {KW_Namespace})
    | ({KW_Module} {WS}+ {KW_Namespace})
    | ({KW_Declare} {WS}+ {KW_Base_uri})
    | {KW_Module}
    ) {NCNonNameChar}*          {
                                    yybegin(NAMESPACE_DECL);
                                    return symbol(SYM_KEYWORD);
                                }
                                
    // Switching to NAMESPACE_KEYWORD                          
    (({KW_Declare} {WS}+ {KW_Default} {WS}+ {KW_Element})
    | ({KW_Default} {WS}+ {KW_Element})
    | ({KW_Declare} {WS}+ {KW_Default} {WS}+ {KW_Function})
    | ({KW_Import} {WS}+ {KW_Schema})
    | ({KW_Import} {WS}+ {KW_Module})
    | ({KW_Declare} {WS}+ {KW_Copy_namespaces})
    ) {NCNonNameChar}*          {
                                    yybegin(NAMESPACE_KEYWORD);
                                    return symbol(SYM_KEYWORD);
                                }
    
    "$"                         {
                                    yypushback(1);
                                    yybegin(VARNAME);
                                }
    (({KW_Declare} {WS}+ {KW_Variable})
    | ({KW_Declare} {WS}+ {KW_Private} {WS}+ {KW_Variable})
    | ({KW_Define} {WS}+ {KW_Variable})
    | {KW_For}
    | {KW_Let}
    | {KW_Some}
    | {KW_Every}) {WS}+ "$"     {
                                    yypushback(1);
                                    return symbol(SYM_KEYWORD);
                                }
    
    ({KW_For} {WS}+ ({KW_Tumbling} | {KW_Sliding}) {WS}+ {KW_Window})
                                {
                                    yypushback(yylength() - 3);
                                    yybegin(WINDOW_CLAUSE);
                                    return symbol(SYM_KEYWORD);
                                }
    
    (({KW_Declare} {WS}+ {KW_Function})
    | ({KW_Define} {WS}+ {KW_Function})
    | ({KW_Declare} {WS}+ {KW_Private} {WS}+ {KW_Function})
    | ({KW_Define} {WS}+ {KW_Private} {WS}+ {KW_Function})
    | ({KW_Declare} {WS}+ {KW_Sequential} {WS}+ {KW_Function})
    | ({KW_Define} {WS}+ {KW_Sequential} {WS}+ {KW_Function})
    | ({KW_Declare} {WS}+ {KW_Ordering} {WS}+ {KW_Ordered})
    | ({KW_Declare} {WS}+ {KW_Ordering} {WS}+ {KW_Unordered})
    ) {NCNonNameChar}*          {
                                    return symbol(SYM_KEYWORD);                                    
                                }
    
    // Global functions and variables.
    ({KW_Declare} {WS}+ ({Annotation} {WS}+)+ ({KW_Function} | {KW_Variable})
    ) {NCNonNameChar}*          {
                                    // Pushback until "declare".
                                    yypushback(yylength() - 7);
                                    pushState(YYINITIAL);
                                    yybegin(ANNOTATION);
                                    return symbol(SYM_KEYWORD);
                                }
    
    (")" {WS}* {KW_As} {NCNonNameChar}*)
                                {
                                    yypushback(yylength() - 1);
                                    yybegin(ITEM_TYPE);
                                    return symbol(SYM_BRACKET);
                                }
    
    {Known_function_name} {WS}* "("
                                {
                                    yypushback(yylength());
                                    pushState(OPERATOR);
                                    yybegin(KNOWN_FUNC);
                                }
                                
    "<!--"                      {
                                    pushState(OPERATOR);
                                    yybegin(XML_COMMENT);
                                    return symbol(SYM_XML_COMMENT);
                                }
    
    "<?"                        {
                                    pushState(OPERATOR);
                                    yybegin(PROCESSING_INSTRUCTION);
                                    return symbol(SYM_PI);
                                }
    
    "<![CDATA["                 {
                                    pushState(OPERATOR);
                                    yybegin(CDATA);
                                    return symbol(SYM_CDATA);
                                }
	{Entity}|{EntityOperator}	{
	                                return symbol(SYM_ENTITY);    
								}                                
                                
    // XML Start Tag
    "<"                         {
                                    pushState(OPERATOR);
                                    yybegin(START_TAG);
                                    return symbol(SYM_XML_TAG);
                                }
    
    ({KW_Declare} {WS}+ {KW_Boundary_space} {NCNonNameChar}*)
                                {
                                    yybegin(XMLSPACE_DECL);
                                    return symbol(SYM_KEYWORD);
                                }
    "}"                         {
                                    yybegin(popState());
                                    return symbol(SYM_CURLY_BRACKET);
                                }
    "{"                         {
                                    pushState(OPERATOR);
                                    return symbol(SYM_CURLY_BRACKET);
                                }
                           
    ({KW_Validate} {WS}* "{")
    | ({KW_Document} {WS}* "{") {
                                    yypushback(yylength() - 8);
                                    return symbol(SYM_KEYWORD);
                                }
                                
    ({KW_Validate} {WS}+ {KW_Lax} {NCNonNameChar}*)
    | ({KW_Validate} {WS}+ {KW_Strict} {NCNonNameChar}*)
                                {
                                    pushState(OPERATOR);
                                    return symbol(SYM_KEYWORD);                                    
                                }
                                
    ({KW_Comment} {WS}* "{")
    | ({KW_Ordered} {WS}* "{") {
                                    yypushback(yylength() - 7);
                                    return symbol(SYM_KEYWORD);
                                }
    
    ({KW_Unordered} {WS}* "{")  {
                                    yypushback(yylength() - 9);
                                    return symbol(SYM_KEYWORD);
                                }

    {KW_Text} {WS}* "{"         {
                                    yypushback(yylength() - 4);
                                    return symbol(SYM_KEYWORD);
                                }
    
    "("                         {   return symbol(SYM_BRACKET);         }
    
    ({KW_Function} "(")         {
                                    yypushback(1);
                                    return symbol(SYM_KEYWORD);
                                }

    // Inline annotated function
    (({Annotation} {WS}+)+ {KW_Function} "(")
                                {
                                    yypushback(yylength());
                                    pushState(YYINITIAL);
                                    yybegin(ANNOTATION);
                                }
    
    {KW_Typeswitch} {WS}* "("   {
                                    yypushback(yylength() - 10);
                                    return symbol(SYM_KEYWORD);
                                }

    {KW_Switch} {WS}* "("       {
                                    yypushback(yylength() - 6);
                                    return symbol(SYM_KEYWORD);
                                }

    "(:"                        {
                                    pushState(YYINITIAL);
                                    yypushback(2);
                                    yybegin(COMMENT);
                                }              
    ({KW_Xquery} {WS}+ {KW_Version} {NCNonNameChar}*)
                                {   
                                    yybegin(XQUERY_VERSION);
                                    return symbol(SYM_KEYWORD);                                    
                                }
    
    "(#"                        {
                                    yybegin(PRAGMA);
                                    return symbol(SYM_PRAGMA);
                                } 
    ({KW_Declare} {WS}+ {KW_Option} {NCNonNameChar}*)
                                {
                                    yybegin(OPTION);
                                    return symbol(SYM_KEYWORD);
                                }
    
    ({KW_Namespace} {NCNonNameChar}*)
                                {   
                                    yybegin(NAMESPACE_DECL);
                                    return symbol(SYM_KEYWORD);
                                }
                                
    ({KW_At} {WS}+ "\"")
    | ({KW_At} {WS}+ "\'")
                                {    
                                    yypushback(yylength() - 2);
                                    yybegin(NAMESPACE_DECL);
                                    return symbol(SYM_KEYWORD);
                                }
                                
    ";"                         {   return symbol(SYM_SEMICOLON);       }
    ","                         {   return symbol(SYM_COMMA);           }
    ({KW_If} {WS}* "(")         {
                                    yypushback(yylength() - 2);
                                    return symbol(SYM_KEYWORD);
                                }
    ({QName} {WS}* "(")         {
                                    yypushback(yylength());
                                    yybegin(UNKNOWN_FUNC);
                                }
    "@" {WS}* {QName}           {   
                                    yybegin(OPERATOR);
                                    return symbol(SYM_ATTRIBUTE_SELECTOR);
                                }
    "@" | "@*"                  {
                                    yybegin(OPERATOR);
                                    return symbol(SYM_ATTRIBUTE_SELECTOR);
                                }
    "-" | "+"                   {   return symbol(SYM_OPERATOR);        }
    "//" | "/"                  {   return symbol(SYM_DELIMITER);       }
    "|" | "!"                   {   return symbol(SYM_OPERATOR);        }
    
    {Axis}                      {   return symbol(SYM_AXIS);            }
    
    {WS}+                       {   return symbol(SYM_TEXT);            }
}

<CATCH_ML> {
    {KW_Catch}                  {   return symbol(SYM_KEYWORD);         }
    {WS}+                       {   return symbol(SYM_TEXT);            }
    "("                         {   
                                    yybegin(VARNAME);
                                    return symbol(SYM_BRACKET);
                                }
}

<CATCH> {
    "*"
	| {QName}
	| ({DQString} ":" "*")      {   return symbol(SYM_IDENTIFIER);      }
	
    {WS}+                       {   return symbol(SYM_TEXT);            }
    
    "("                         {   
                                    yybegin(VARNAME);
                                    return symbol(SYM_BRACKET);
                                }
                                
    "{"                         {   
                                    yypushback(1);
                                    yybegin(YYINITIAL);
                                }
                                
    "|"                         {   return symbol(SYM_OPERATOR);        }
}

// Predefined functions.
<KNOWN_FUNC> {
    {Known_function_name} {WS}+ "(:"
                                {
                                    yypushback(yylength());
                                    // Get rid of the stored state...
                                    popState();
                                    yybegin(FAKE_FUNC);
                                }
    
    {Known_function_name}       {   return symbol(SYM_FUNCTION);        }
    
    {WS}+                       {   return symbol(SYM_TEXT);            }
    "("                         {   
                                    yybegin(KINDTEST);
                                    return symbol(SYM_BRACKET);
                                }     
}

<UNKNOWN_FUNC> {
    {QName} {WS}+ "(:"          {
                                    yypushback(yylength());
                                    yybegin(FAKE_FUNC);
                                }
    {QName}                     {   return symbol(SYM_FUNCTION);        }
    {WS}+                       {   return symbol(SYM_TEXT);            }
    "("                         {   
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_BRACKET);
                                }
}

<FAKE_FUNC> {
    {QName}                     {   return symbol(SYM_IDENTIFIER);      }
    {WS}+                       {   return symbol(SYM_TEXT);            }
    "(:"                        {   
                                    pushState(YYINITIAL);
                                    yypushback(2);
                                    yybegin(COMMENT);
                                }
}

<XQUERY_VERSION> {
    {KW_Encoding} {NCNonNameChar}*
                                {   return symbol(SYM_KEYWORD);         }
    
    {DQString}                  {   return symbol(SYM_STRING_DQ);       }
    {DQUnclosedString}          {   
                                    pushState(XQUERY_VERSION);
                                    yybegin(DQ_STRING_MULTILINE);   
                                    return symbol(SYM_STRING_DQ);       
                                }
    {SQString}                  {   return symbol(SYM_STRING_SQ);       }
    {SQUnclosedString}          {   
                                    pushState(XQUERY_VERSION);
                                    yybegin(SQ_STRING_MULTILINE);
                                    return symbol(SYM_STRING_SQ);       
                                }
    {WS}+                       {   return symbol(SYM_TEXT);            }
    ";"                         {   
                                    yybegin(YYINITIAL);   
                                    return symbol(SYM_SEMICOLON);       
                                }
    
    <<EOF>>                     {   yybegin(YYINITIAL);                 }                            
}

<COMMENT> {
    "(:"                        {
                                    cLen += 2;
                                    incrementNestingLevel();
                                }
    ":)"                        {
                                    cLen += 2;
                                    decrementNestingLevel();
                                    if (getNestingLevel() == 0) {
	                                    yybegin(popState());
	                                    return flush(SYM_COMMENT);
                                    }
                                }
                                
    {Char}                      {   cLen++;                             }
    <<EOF>>                     {   
                                    return flush(SYM_COMMENT);
                                }
}

<DQ_STRING_MULTILINE> {
    "\""                        {  
                                    cLen++;
                                    yybegin(popState());
                                    return flush(SYM_STRING_DQ);
                                }
    {Char}                      {   cLen++;    }
    "\"\""                      {   cLen += 2; }
    <<EOF>>                     {
                                    return flush(SYM_STRING_DQ);
                                }
}

<SQ_STRING_MULTILINE> {
    "\'"                        {
                                    cLen++;
                                    yybegin(popState());
                                    return flush(SYM_STRING_SQ);
                                }
    {Char}                      {   cLen++;    }
    "\'\'"                      {   cLen += 2; }
    <<EOF>>                     {
                                    return flush(SYM_STRING_SQ);
                                }
}

<OPERATOR> 
{
    "<!--"                      {
                                    pushState(OPERATOR);
                                    yybegin(XML_COMMENT);
                                    return symbol(SYM_XML_COMMENT);
                                }
    
    "<?"                        {
                                    pushState(OPERATOR);
                                    yybegin(PROCESSING_INSTRUCTION);
                                    return symbol(SYM_PI);
                                }
    
    "<![CDATA["                 {
                                    pushState(OPERATOR);
                                    yybegin(CDATA);
                                    return symbol(SYM_CDATA);
                                }
                                
    // XML Start Tag
    ("<" {QName})               {
                                    yypushback(yylength() - 1);
                                    pushState(OPERATOR);
                                    yybegin(START_TAG);
                                    return symbol(SYM_XML_TAG);
                                }
    "{"                         {
                                    pushState(OPERATOR);
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_CURLY_BRACKET);
                                }
    "}"                         {
                                    yybegin(popState());
                                    return symbol(SYM_CURLY_BRACKET);
                                }                                
    "(:"                        {
                                    pushState(OPERATOR);
                                    yypushback(2);
                                    yybegin(COMMENT);
                                }
    ";"                         {
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_SEMICOLON);
                                }
    ({KW_And}
    | {KW_Div}
    | {KW_Mod}
    ) {NCNonNameChar}*          {  
                                    yypushback(yylength() - 3);
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_OPERATOR); 
                                }
    ({KW_Idiv} {NCNonNameChar}*) {  
                                    yypushback(yylength() - 4);
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_OPERATOR); 
                                }
    ({KW_Eq}
    | {KW_Ge}
    | {KW_Gt}
    | {KW_Le}
    | {KW_Lt}
    | {KW_Ne}
    | {KW_Is}
    | {KW_Or}) {NCNonNameChar}* {  
                                    yypushback(yylength() - 2);
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_OPERATOR); 
                                }
    ":="
    | "="
    | ">="
    | ">>"
    | ">"
    | "<="
    | "<<"
    | "<"
    | "-"
    | "*"
    | "!="
    | "+"
    | "||"
    | "|"
    | {Entity}
    | {EntityOperator}          {  
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_OPERATOR); 
                                }
    "//"
    | "/"                       {   
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_DELIMITER);
                                }
    ","                         {   
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_COMMA);
                                }
      
    "["                         {
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_SQUARE_BRACKET);  
                                }

    ({KW_Default} {WS}+ {KW_Return})
                                {
                                    // Push back and emit only "default". "return" will be emmitted bellow.
                                    yypushback(yylength() - 7);
                                    return symbol(SYM_KEYWORD);
                                }
    
    ({KW_Then}
    | {KW_Count}
    | {KW_Case}
    | {KW_Else}
    | {KW_External}
    | {KW_Except}
    | {KW_Intersect}
    | {KW_In}
    | {KW_At}
    | ({KW_Allowing} {WS}+ {KW_Empty})
    | ({KW_Group} {WS}+ {KW_By})
    | ({KW_Order} {WS}+ {KW_By})
    | ({KW_Stable} {WS}+ {KW_Order} {WS}+ {KW_By})
    | {KW_Return}
    | {KW_Satisfies}
    | {KW_To}
    | {KW_Union}
    | {KW_Where}
    | {KW_Preserve}
    | {KW_Strip}) {NCNonNameChar}*
                                {   
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_KEYWORD); 
                                }
                                
    (({KW_Castable} | {KW_Cast}) {WS}+ {KW_As} {NCNonNameChar}*
    )                           {
                                    yybegin(SINGLE_TYPE);
                                    return symbol(SYM_KEYWORD);
                                }  
      
    (({KW_Instance} {WS}+ {KW_Of})
    | ({KW_Treat} {WS}+ {KW_As})
    | {KW_Case}
    | {KW_As}) {NCNonNameChar}* {
                                    yybegin(ITEM_TYPE);
                                    return symbol(SYM_KEYWORD);
                                }
                                
    {WS}+                       {   return symbol(SYM_TEXT);            }                            
    
    ({KW_For} {WS}+ ({KW_Tumbling} | {KW_Sliding}) {WS}+ {KW_Window})
                                {
                                    yypushback(yylength() - 3);
                                    yybegin(WINDOW_CLAUSE);
                                    return symbol(SYM_KEYWORD);
                                }
    
    
    // "let" & "for" are accepted as key words only if followed by "$" 
    // We'll treat "$" separatelly.
    ({KW_For} | ({KW_Let}) {WS}+ "$")
                                {
                                    yypushback(yylength() - 3);
                                    return symbol(SYM_KEYWORD); 
                                }
    (({KW_Start} | ({KW_Start} {WS}+ {KW_At})
    | (({KW_Only} {WS}+)? ({KW_End} | ({KW_End} {WS}+ {KW_At})))
    ) {WS}+ "$")
                                {
                                    // Pushback only the $, emit the rest as a keyword.
                                    yypushback(1);
                                    return symbol(SYM_KEYWORD); 
                                }
    ({KW_Previous} {WS}+ "$")   {
                                    yypushback(yylength() - 8);
                                    return symbol(SYM_KEYWORD); 
                                }
    ({KW_Next} {WS}+ "$")       {
                                    yypushback(yylength() - 4);
                                    return symbol(SYM_KEYWORD); 
                                }
    ({KW_At} {WS}+ "$")         {
                                    yypushback(yylength() - 2);
                                    return symbol(SYM_KEYWORD); 
                                }
    {KW_When}                   {   return symbol(SYM_KEYWORD);         }
     
    "$"                         {
                                    yypushback(1);
                                    yybegin(VARNAME);
                                }
    ({KW_Function} "(")         {
                                    yypushback(1);
                                    return symbol(SYM_KEYWORD);
                                }
                                
    // Inline annotated function
    (({Annotation} {WS}+)+ {KW_Function} "(")
                                {
                                    yypushback(yylength());
                                    pushState(OPERATOR);
                                    yybegin(ANNOTATION);
                                }
                                
    "("                         {   return symbol(SYM_BRACKET);         }
    ")"                         {   return symbol(SYM_BRACKET);         }
    "?"                         {   return symbol(SYM_ASK);             }
    "]"                         {   return symbol(SYM_SQUARE_BRACKET);  }
    
    (({KW_Empty} {WS}+ {KW_Greatest})
    | ({KW_Empty} {WS}+ {KW_Least})
    | {KW_Ascending}
    | {KW_Descending}
    | {KW_Collation}
    | {KW_Default}
    ) {NCNonNameChar}*          {
                                    return symbol(SYM_KEYWORD);  
                                }
    
    {DQString}                  {   return symbol(SYM_STRING_DQ);     }
    {DQUnclosedString}          {   
                                    pushState(OPERATOR);
                                    yybegin(DQ_STRING_MULTILINE);   
                                    return symbol(SYM_STRING_DQ);       
                                }
    
    {SQString}                  {   return symbol(SYM_STRING_SQ);     }
    
    {SQUnclosedString}          {   
                                    pushState(OPERATOR);
                                    yybegin(SQ_STRING_MULTILINE);
                                    return symbol(SYM_STRING_SQ);       
                                }
    {NCName}                    {
                                    // This is a fallback...   
                                    yypushback(yylength());
                                    yybegin(YYINITIAL);
                                }
    [^;,=<>-*+/\[\]\)|?${}]     {
                                    // This is a fallback...   
                                    yypushback(1);
                                    yybegin(YYINITIAL);
                                }
    
}

<SINGLE_TYPE> {
    {QName}                     {   
                                    yybegin(OPERATOR);
                                    return symbol(SYM_IDENTIFIER);      
                                }
    
    {WS}+                       {   return symbol(SYM_TEXT);            }
    
    "(:"                        {
                                    yypushback(2);
                                    pushState(SINGLE_TYPE);
                                    yybegin(COMMENT);
                                }
                                
    {Char}                      {
                                    yypushback(1);
                                    yybegin(popState());
                                }
}

<ITEM_TYPE> {
    "$"                         {
                                    yypushback(1);
                                    yybegin(VARNAME);
                                }
    
    "(:"                        {
                                    yypushback(2);
                                    pushState(ITEM_TYPE);
                                    yybegin(COMMENT);
                                }
        
    ({KW_Element}
     | {KW_Attribute}
     | {KW_Schema_element}
     | {KW_Schema_attribute}
     | {KW_Comment}
     | {KW_Text}
     | {KW_Processing_instruction}
     | {KW_Node}
     | {KW_Document_node}
    ) {WS}* "("                 {
                                    yypushback(yylength());
                                    pushState(OCCURENCE_INDICATOR);
                                    yybegin(KNOWN_FUNC);
                                }
    
    "("                         {
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_BRACKET);
                                }
                                
    {KW_Item} {WS}* "(" {WS}* ")" 
                                {
                                    yypushback(yylength());
                                    yybegin(ITEM);
                                }

    {KW_Void} {WS}* "(" {WS}* ")" 
                                {
                                    yypushback(yylength());
                                    yybegin(VOID);
                                }


    "(#"                        {
                                    yybegin(PRAGMA);
                                    return symbol(SYM_PRAGMA);
                                }

    ";"                         {
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_SEMICOLON);
                                }
    
    ({KW_Then} | {KW_Else}) {NCNonNameChar}*
                                {
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_KEYWORD);
                                }
    
    ({KW_At} {WS}+ "\"")
    | ({KW_At} {WS}+ "\'")
                                {    
                                    yypushback(yylength() - 2);
                                    yybegin(NAMESPACE_DECL);
                                    return symbol(SYM_KEYWORD);
                                }

    ({KW_External} | {KW_At} 
    | {KW_Except} | {KW_Intersect} 
    | {KW_In} | {KW_Is} 
    | {KW_Return} | {KW_Satisfies} 
    | {KW_To} | {KW_Union} 
    | {KW_Where} 
    | {KW_Count}
    | ({KW_Order} {WS}+ {KW_By}) 
    | ({KW_Stable} {WS}+ {KW_Order} {WS}+ {KW_By})
    ) {NCNonNameChar}*          {
                                    
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_KEYWORD);
                                }
     
    (({KW_And} | {KW_Div} 
    | {KW_Eq} | {KW_Ge} 
    | {KW_Gt} | {KW_Le} 
    | {KW_Lt} | {KW_Ne} 
    | {KW_Idiv} | {KW_Mod} 
    | {KW_Or}) {NCNonNameChar}*)
    | ":=" 
    | ">=" | ">>" 
    | ">" | "<=" 
    | "<<" | "<" 
    | "-" | "!=" 
    | "|" | "="                 {
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_OPERATOR);
                                }
    
    ","                         {
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_COMMA);
                                }
    "["                         {
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_SQUARE_BRACKET);
                                }
    (({KW_Castable} {WS}+ {KW_As})
    | ({KW_Cast} {WS}+ {KW_As})
    ) {NCNonNameChar}*          {
                                    yybegin(SINGLE_TYPE);
                                    return symbol(SYM_KEYWORD);
                                }
    
    (({KW_Instance} {WS}+ {KW_Of})
    | ({KW_Treat} {WS}+ {KW_As})
    | {KW_Case}
    | {KW_As}) {NCNonNameChar}* {   return symbol(SYM_KEYWORD);         }
    
    // Inline annotated function
    (({Annotation} {WS}+)* {KW_Function} "(")
                                {
                                    yypushback(yylength());
                                    pushState(YYINITIAL);
                                    yybegin(ANNOTATION);
                                }
    
    ")"                         {   return symbol(SYM_BRACKET);         }
    
    // Strings
    {DQString}                  {   return symbol(SYM_STRING_DQ);       }
    {DQUnclosedString}          {   
                                    pushState(KINDTEST);
                                    yybegin(DQ_STRING_MULTILINE);   
                                    return symbol(SYM_STRING_DQ);       
                                }
    
    {SQString}                  {   return symbol(SYM_STRING_SQ);       }
    
    {SQUnclosedString}          {   
                                    pushState(KINDTEST);
                                    yybegin(SQ_STRING_MULTILINE);
                                    return symbol(SYM_STRING_SQ);       
                                }

    {QName}                     {
                                    yybegin(OCCURENCE_INDICATOR);
                                    return symbol(SYM_IDENTIFIER);
                                }
    {WS}+                       {   return symbol(SYM_TEXT);            }
}

<VARNAME> {
    {WS}+                       {   return symbol(SYM_TEXT);            }
    "$" {QName}                 {
                                    yybegin(OPERATOR);
                                    return symbol(SYM_VARIABLE);
                                }
    "$"                         {  // Fallback for editing purposes.
                                    yybegin(OPERATOR);
                                    return symbol(SYM_VARIABLE);
                                }
    "(:"                        {
                                    yypushback(2);
                                    pushState(VARNAME);
                                    yybegin(COMMENT);
                                }
}

<ITEM> {
    {KW_Item}                   {   return symbol(SYM_KEYWORD);         }
    "("                         {   return symbol(SYM_BRACKET);         }
    ")"                         {
                                    yybegin(OCCURENCE_INDICATOR);
                                    return symbol(SYM_BRACKET);
                                }
    {WS}+                       {   return symbol(SYM_TEXT);            }
}

<VOID> {
    {KW_Void}                   {   return symbol(SYM_KEYWORD);         }
    "("                         {   return symbol(SYM_BRACKET);         }
    ")"                         {
                                    yybegin(OPERATOR);
                                    return symbol(SYM_BRACKET);
                                }
    {WS}+                       {   return symbol(SYM_TEXT);            }
}

<KINDTEST> {
    "{"                         {
                                    pushState(OPERATOR);
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_CURLY_BRACKET);
                                }
    ")"                         {
                                    yybegin(popState());
                                    return symbol(SYM_BRACKET);
                                }
    ","                         {   return symbol(SYM_COMMA);               }
    "?"                         {   return symbol(SYM_ASK);                 }
    "*"                         {   return symbol(SYM_STAR);                }
    "(:"                        {
                                    yypushback(2);
                                    pushState(KINDTEST);
                                    yybegin(COMMENT);
                                }
    
    ({KW_Element} {WS}* "(")
    | ({KW_Schema_element} {WS}* "(")
                                {
                                    yypushback(yylength());
                                    pushState(KINDTEST);
                                    yybegin(KNOWN_FUNC);
                                }
    {QName}                     {   return symbol(SYM_IDENTIFIER);          }
    
    // Strings
    {DQString}                  {   return symbol(SYM_STRING_DQ);           }
    {DQUnclosedString}          {   
                                    pushState(KINDTEST);
                                    yybegin(DQ_STRING_MULTILINE);   
                                    return symbol(SYM_STRING_DQ);       
                                }
    
    {SQString}                  {   return symbol(SYM_STRING_SQ);           }
    
    {SQUnclosedString}          {   
                                    pushState(KINDTEST);
                                    yybegin(SQ_STRING_MULTILINE);
                                    return symbol(SYM_STRING_SQ);       
                                }
    {WS}+                       {   return symbol(SYM_TEXT);                }
}

<OCCURENCE_INDICATOR> {
    "?"                         {   
                                    yybegin(OPERATOR);
                                    return symbol(SYM_ASK);
                                }
    "+"                         {   
                                    yybegin(OPERATOR);
                                    return symbol(SYM_PLUS);
                                }
    "*"                         {   
                                    yybegin(OPERATOR);
                                    return symbol(SYM_STAR);
                                }
    "(:"                        {
                                    yypushback(2);
                                    pushState(OCCURENCE_INDICATOR);
                                    yybegin(COMMENT);
                                }
                                
    [^\?\+\*]                   {  
                                    yypushback(1); 
                                    yybegin(OPERATOR);
                                }
}

<PRAGMA> {
    {QName}                     {   
                                    yybegin(PRAGMA_CONTENTS);
                                    return symbol(SYM_IDENTIFIER);
                                }
    {WS}+                       {   return symbol(SYM_TEXT);                    }
}

<PRAGMA_CONTENTS> {
    "#)"                        {
                                    yybegin(OPERATOR);
                                    return symbol(SYM_PRAGMA);
                                }
    ~"#)"                       {
                                    yypushback(2);
                                    return symbol(SYM_PRAGMA_CONTENT);
                                }
    {Char}                      {   cLen++;                                     }
    <<EOF>>                     {   return flush(SYM_PRAGMA_CONTENT);           }
}

<NAMESPACE_DECL> {
    ({KW_At} {WS}+ "\"")
    | ({KW_At} {WS}+ "\'")      {    
                                    yypushback(yylength() - 2);
                                    return symbol(SYM_KEYWORD);
                                }
                                
    {WS}+                       {   return symbol(SYM_TEXT);                    }
    
    // After strings the end of the namespace declaration is expected...
    {DQString}                  {   
                                    yybegin(NAMESPACE_END_EXPECTED);
                                    return symbol(SYM_STRING_DQ);
                                }
    {DQUnclosedString}          {   
                                    pushState(NAMESPACE_END_EXPECTED);
                                    yybegin(DQ_STRING_MULTILINE);   
                                    return symbol(SYM_STRING_DQ);       
                                }
    
    {SQString}                  {   
                                    yybegin(NAMESPACE_END_EXPECTED);
                                    return symbol(SYM_STRING_SQ);
                                }
    
    {SQUnclosedString}          {   
                                    pushState(NAMESPACE_END_EXPECTED);
                                    yybegin(SQ_STRING_MULTILINE);
                                    return symbol(SYM_STRING_SQ);       
                                }

    {NCName}                    {   return symbol(SYM_IDENTIFIER);              }
    "="                         {   return symbol(SYM_EQUAL);                   }
    ","                         {   return symbol(SYM_COMMA);                   }
    ";"                         {
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_SEMICOLON);                   
                                }
    "(:"                        {
                                    yypushback(2);
                                    pushState(NAMESPACE_DECL);
                                    yybegin(COMMENT);
                                }
    "{"                         {
                                    pushState(NAMESPACE_DECL);
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_CURLY_BRACKET);
                                }
    <<EOF>>                     {   yybegin(YYINITIAL);                         }
}

<NAMESPACE_END_EXPECTED>        {
    ({KW_At} {WS}+ "\"")
    | ({KW_At} {WS}+ "\'")      {    
                                    yypushback(yylength() - 2);
                                    yybegin(NAMESPACE_DECL);
                                    return symbol(SYM_KEYWORD);
                                }
    {WS}+                       {   return symbol(SYM_TEXT);                    }
    ","                         {
                                    yybegin(NAMESPACE_DECL);
                                    return symbol(SYM_COMMA);
                                }
    ";"                         {
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_SEMICOLON);                   
                                }
    <<EOF>>                     {   yybegin(YYINITIAL);                         }                                
}

<NAMESPACE_KEYWORD> {
    {WS}+                       {   return symbol(SYM_TEXT);                    }
    // Strings
    {DQString}                  {
                                    yybegin(NAMESPACE_DECL);
                                    return symbol(SYM_STRING_DQ);               
                                }
    {DQUnclosedString}          {   
                                    pushState(NAMESPACE_KEYWORD);
                                    yybegin(DQ_STRING_MULTILINE);   
                                    return symbol(SYM_STRING_DQ);       
                                }
    {SQString}                  {
                                    yybegin(NAMESPACE_DECL);
                                    return symbol(SYM_STRING_SQ);               
                                }
    {SQUnclosedString}          {   
                                    pushState(NAMESPACE_KEYWORD);
                                    yybegin(SQ_STRING_MULTILINE);
                                    return symbol(SYM_STRING_SQ);       
                                }
    ({KW_Inherit}
    | {KW_No_inherit}) {NCNonNameChar}*
                                {   
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_KEYWORD);
                                }
    ({KW_Namespace} {NCNonNameChar}*)
                                {   
                                    yybegin(NAMESPACE_DECL);
                                    return symbol(SYM_KEYWORD);
                                }
    (({KW_Default} {WS}+ {KW_Element})
    | {KW_No_preserve}
    | {KW_Preserve}) {NCNonNameChar}*
                                {   
                                    return symbol(SYM_KEYWORD);
                                }
    ","                         {   return symbol(SYM_COMMA);                   }
    "(:"                        {
                                    yypushback(2);
                                    pushState(NAMESPACE_KEYWORD);
                                    yybegin(COMMENT);
                                }
}

<XML_COMMENT> {
    "-->"                       {
                                    yybegin(popState());
                                    return symbol(SYM_XML_COMMENT);
                                }
    ~"-->"                      {
                                    yypushback(3);
                                    return symbol(SYM_XML_COMMENT);
                                }
    {Char}                      {   cLen++;                                     }
    <<EOF>>                     {   return flush(SYM_XML_COMMENT);              }
}

<PROCESSING_INSTRUCTION> {
    "?>"                        {
                                    yybegin(popState());
                                    return symbol(SYM_PI);
                                }
    ~"?>"                       {
                                    yypushback(2);
                                    return symbol(SYM_PI);
                                }
    {Char}                      {   cLen++;                                     }
    <<EOF>>                     {   return flush(SYM_PI);                       }
}

<CDATA> {
    "]]>"                       {
                                    yybegin(popState());
                                    return symbol(SYM_CDATA);
                                }
    ~"]]>"                      {
                                    yypushback(3);
                                    return symbol(SYM_CDATA);
                                }
    {Char}                      {   cLen++;                                     }
    <<EOF>>                     {   return flush(SYM_CDATA);                    }
}

<START_TAG> {
    "/>"                        {
                                    yybegin(popState());
                                    return symbol(SYM_XML_TAG);
                                }
    ">"                         {
                                    yybegin(ELEMENT_CONTENT);
                                    return symbol(SYM_XML_TAG);
                                }
    {WS}+                       {   return symbol(SYM_TEXT);                    }
     
    {QName}                     {   
                                    yybegin(XML_ATTRIBUTES);
                                    return symbol(SYM_XML_ELEMENT);             
                                }
}

<XML_ATTRIBUTES> {
    "/>"                        {
                                    yybegin(popState());
                                    return symbol(SYM_XML_TAG);
                                }
    ">"                         {
                                    yybegin(ELEMENT_CONTENT);
                                    return symbol(SYM_XML_TAG);
                                }
                                
    {WS}+                       {   return symbol(SYM_TEXT);                    }
     
    {QName}                     {   return symbol(SYM_XML_ATTRIBUTE);           }
    "="                         {   return symbol(SYM_EQUAL);                   }
    "\""                        {   
                                    yybegin(QUOT_ATTRIBUTE_CONTENT);
                                    return symbol(SYM_ATTRIBUTE_VALUE);
                                }
    "\'"                        {   
                                    yybegin(APOS_ATTRIBUTE_CONTENT);
                                    return symbol(SYM_ATTRIBUTE_VALUE);
                                }
    "{"                         {
                                    pushState(XML_ATTRIBUTES);
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_CURLY_BRACKET);
                                }
}

<QUOT_ATTRIBUTE_CONTENT> {
    "\""                        {
                                    yybegin(XML_ATTRIBUTES);
                                    return symbol(SYM_ATTRIBUTE_VALUE);
                                }
    "{"                         {
                                    pushState(QUOT_ATTRIBUTE_CONTENT);
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_CURLY_BRACKET);
                                }
                                    
    ([^\"\{\}<]
    | "\"\"" 
    | "{{"
    | "}}")+                    {   return symbol(SYM_ATTRIBUTE_VALUE);         }
}

<APOS_ATTRIBUTE_CONTENT> {
    "\'"                        {
                                    yybegin(XML_ATTRIBUTES);
                                    return symbol(SYM_ATTRIBUTE_VALUE);
                                }
    "{"                         {
                                    pushState(APOS_ATTRIBUTE_CONTENT);
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_CURLY_BRACKET);
                                }
           
    ([^\'\{\}<]
    | "\'\'" 
    | "{{"
    | "}}")+                    {   return symbol(SYM_ATTRIBUTE_VALUE);         }
}

<ELEMENT_CONTENT> {
    
    "</"                        {
                                    yybegin(END_TAG);
                                    return symbol(SYM_XML_TAG);
                                }
    "{"                         {
                                    pushState(ELEMENT_CONTENT);
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_CURLY_BRACKET);
                                }
                                
    "<!--"                      {
                                    pushState(ELEMENT_CONTENT);
                                    yybegin(XML_COMMENT);
                                    return symbol(SYM_XML_COMMENT);
                                }
    
    "<?"                        {
                                    pushState(ELEMENT_CONTENT);
                                    yybegin(PROCESSING_INSTRUCTION);
                                    return symbol(SYM_PI);
                                }
    
    "<![CDATA["                 {
                                    pushState(ELEMENT_CONTENT);
                                    yybegin(CDATA);
                                    return symbol(SYM_CDATA);
                                }
	{Entity}					{
	                                return symbol(SYM_ENTITY);    
								}                                
    // XML Start Tag
    "<"                         {
                                    pushState(ELEMENT_CONTENT);
                                    yybegin(START_TAG);
                                    return symbol(SYM_XML_TAG);
                                }                            
    ([^\{\}<] 
     | "{{" 
     | "}}")+                   {   return symbol(SYM_TEXT);                   }
           
}

<END_TAG> {
    ">"                        {
                                    yybegin(popState());
                                    return symbol(SYM_XML_TAG);
                                }
    {WS}+                       {   return symbol(SYM_TEXT);                    }
     
    {QName}                     {   return symbol(SYM_XML_ELEMENT);             }
}

<XMLSPACE_DECL> {
    ({KW_Preserve}
    | {KW_Strip}) {NCNonNameChar}*
                                {   
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_KEYWORD);
                                }
    "(:"                        {
                                    pushState(XMLSPACE_DECL);
                                    yypushback(2);
                                    yybegin(COMMENT);
                                }
}

<OPTION> {
    {QName}                     {
                                    yybegin(YYINITIAL);
                                    return symbol(SYM_IDENTIFIER);
                                }
    {WS}+                       {   return symbol(SYM_TEXT);                    }
}

<ANNOTATION> {
    {Annotation}                {   return symbol(SYM_ANNOTATION);              }
    {KW_Variable}               {
                                    yybegin(popState());
                                    return symbol(SYM_KEYWORD);
                                }
    {KW_Function}               {
                                    yybegin(popState());
                                    return symbol(SYM_KEYWORD);
                                }
    {WS}+                       {   return symbol(SYM_TEXT);                    }
}

<WINDOW_CLAUSE> {
    {KW_Tumbling}               {   return symbol(SYM_KEYWORD);                 }
    {KW_Sliding}                {   return symbol(SYM_KEYWORD);                 }
    {KW_Window}                 {   return symbol(SYM_KEYWORD);                 }
    {WS}+                       {   return symbol(SYM_TEXT);                    }
    "$"                         {
                                    yypushback(1);
                                    yybegin(VARNAME);
                                }
}