<?php
/**
 * Validator class
 * @author Jérôme Brilland
 * @version 2.0.1
 * @copyright (c) 2009-2022, Jérôme Brilland
 * @license http://www.gnu.org/licenses/gpl.txt GNU General Public License
 */
class Validator {

    /**
     * Checks if a value is not empty.
     * @param string value Value
     * @return boolean True if the value is not empty, false otherwise
     * @access public
     * @static
     */
    public static function validateNotEmpty( $value ) {
        if ( 0 == strlen( trim( $value ) ) ) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Checks if a value contains only letters.
     * @param string value Value
     * @return boolean True if the value contains only letters, false otherwise
     * @access public
     * @static
     */
    public static function validateLettersOnly( $value ) {
        if ( !preg_match( '/^[A-Za-z -]*$/im', $value ) ) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Checks if a value contains only letters and numbers.
     * @param string value Value
     * @return boolean True if the value contains only letters and digits, false otherwise
     * @access public
     * @static
     */
    public static function validateAlphanumeric( $value ) {
        if ( !preg_match( '/^[A-Za-z0-9 -]*$/im', $value ) ) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Checks if a value's length is lesser than or equal to the maximum length.
     * @param string value Value
     * @param int max Maximum length
     * @return boolean True if the input length is lesser than or equal to the maximum length, false otherwise
     * @access public
     * @static
     */
    public static function validateMaxLength( $value, $max ) {
        if ( mb_strlen( $value, 'UTF-8' ) > $max ) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Checks if a value's length is greater than or equal to the minimum value.
     * @param string value Value
     * @param int min Minimum length
     * @return boolean True if the input length is greater than or equal to the minimum length, false otherwise
     * @access public
     * @static
     */
    public static function validateMinLength( $value, $min ) {
        if ( mb_strlen( $value, 'UTF-8' ) < $min ) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Checks if the given value is a valid number.
     * @param string value Value
     * @return boolean True if the value is a valid number, false otherwise
     * @access public
     * @static
     */
    public static function validateNumber( $value ) {
        return is_numeric( $value );
    }

    /**
     * Checks if the value is a valid integer.
     * @param string value Value
     * @return boolean True if the value is a valid integer, false otherwise
     * @access public
     * @static
     */
    public static function validateInteger( $value ) {
        if ( filter_var( $value, FILTER_VALIDATE_INT ) === false ) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Checks if a value is greater than or equal to a minimum value.
     * @param string value Value
     * @param int min Minimum value
     * @return boolean True if the value is greater than or equal to the minimum value, false otherwise
     * @access public
     * @static
     */
    public static function validateMin( $value, $min ) {
        if ( $value < $min ) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Checks if a value is less than or equal to a maximum value.
     * @param string value Value
     * @param int max Maximum value
     * @return boolean True if the value is less than or equal to the maximum value, false otherwise
     * @access public
     * @static
     */
    public static function validateMax( $value, $max ) {
        if ( $value > $max ) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Checks if a value is a valid date.
     * @param string value Value
     * @return boolean True if the value is a valid date, false otherwise
     * @access public
     * @static
     */
    public static function validateDate( $value ) {
        if ( is_array( $value ) ) {
            return checkdate( $value['month'], $value['day'], $value['year'] );
        } else {
            $dateExploded = explode( '-', $value );
            if ( count( $dateExploded ) != 3 ) {
                return false;
            } else if ( ( !is_numeric( $dateExploded[0] ) ) || ( !is_numeric( $dateExploded[1] ) ) || ( !is_numeric( $dateExploded[2] ) ) ) {
                return false;
            } else {
                list( $year, $month, $day ) = $dateExploded;
                return checkdate( $month, $day, $year );
            }
        }
    }

    /**
     * checks if a value is a valid time.
     * @param string value Value
     * @return boolean True if the value is a valid time, false otherwise
     * @access public
     * @static
     */
    public static function validateTime( $value ) {
        if ( !preg_match( '/^([0-1][0-9]|2[0-3]):([0-5][0-9])$/', $value ) && !preg_match( '/^([0-1][0-9]|2[0-3]):([0-5][0-9]):([0-5][0-9])$/', $value ) ) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Checks if the value is a valid URL.
     * @param string value Value
     * @return boolean True if the value is a valid URL, false otherwise
     * @access public
     * @static
     */
    public static function validateURL( $value ) {
        if ( !preg_match( '%^(?:(?:https?|ftp)://)(?:\S+(?::\S*)?@|\d{1,3}(?:\.\d{1,3}){3}|(?:(?:[a-z\d\x{00a1}-\x{ffff}]+-?)*[a-z\d\x{00a1}-\x{ffff}]+)(?:\.(?:[a-z\d\x{00a1}-\x{ffff}]+-?)*[a-z\d\x{00a1}-\x{ffff}]+)*(?:\.[a-z\x{00a1}-\x{ffff}]{2,6}))(?::\d+)?(?:[^\s]*)?$%iu', $value ) ) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Checks if the value is a valid email address.
     * @param string value Value
     * @return boolean True if the value is a valid email address, false otherwise
     * @access public
     * @static
     */
    public static function validateEmail( $value ) {
        if ( !preg_match( '#^[a-z0-9._-]+@[a-z0-9.-]{2,}\.[a-z]{2,4}$#i', $value ) ) {
            return false;
        } elseif ( !checkdnsrr( substr( strrchr( $value, '@' ), 1 ), 'MX' ) ) {
            return false;
        }
        return true;
    }

    /**
     * Checks if the value is a valid IP address.
     * @param string value Value
     * @return boolean True if the value is a valid IP address, false otherwise
     * @access public
     * @static
     */
    public static function validateIP( $value ) {
        if ( filter_var( $value, FILTER_VALIDATE_IP ) === false ) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Compares two value and returns true id they are equal.
     * @param string value1 First value
     * @param string value2 Second value
     * @return boolean True if the two values are equal, false otherwise
     * @access public
     * @static
     */
    public static function validateEqual( $value1, $value2 ) {
        if ( $value1 != $value2 ) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Checks if a value exists in an array
     * @param string value Value
     * @param string array Array
     * @return boolean
     * @access protected
     */
    public static function validateInArray( $value, $array ) {
        return in_array( $value, $array );
    }

    /**
     * Checks if the given value match the regular expression.
     * @param string value Value
     * @param string pattern Regular expression
     * @return boolean True if the value matches the regular expression, false otherwise
     * @access public
     * @static
     */
    public static function validateRegExpr( $value, $pattern ) {
        if ( !preg_match( $pattern, $value ) ) {
            return false;
        } else {
            return true;
        }
    }

}


