<?php
/**
 * Request class
 * @author Jérôme Brilland
 * @version 2.2
 * @copyright (c) 2009-2022, Jérôme Brilland
 * @license http://www.gnu.org/licenses/gpl.txt GNU General Public License 
 */
class Request {
    
    /**
     * @var string $domain
     * @access private
     */
    private $domain; 

    /**
     * @var string $method Request method
     * @access private
     */
    private $method;

    /**
     * @var string $uri Request URI
     * @access private
     */
    private $uri;
    
    /**
     * @var string $path
     * @access private
     */
    private $path;
    
    /**
     * @var array $request Request
     * @access private
     */
    private $request;
    
    /**
     * @var array $cookies
     * @access private
     */
    private $cookies; 

    /**
     * @var array $attributes Attributes derived from the request
     * @access protected
     */
    protected $attributes = [];

    /**
     * Class constructor
     * @access public
     */
    public function __construct() {
        $this->domain = $_SERVER['HTTP_HOST'];
        $this->method = $_SERVER['REQUEST_METHOD'];
        $this->uri = $_SERVER['REQUEST_URI'];
        $this->path = explode( '?', $_SERVER['REQUEST_URI'] )[0];
        $this->request = new FilteredMap( array_merge( $_POST, $_GET ) );
        $this->cookies = new FilteredMap( $_COOKIE );
    }
    
    /**
     * @return string
     * @access public
     */
    public function getDomain() {
        return $this->domain;
    }

    /**
     * Retrieve the HTTP method of the request.
     * @return string the request method
     * @access public
     */
    public function getMethod() {
        return $this->method;
    }

    /**
     * @return string
     * @access public
     */
    public function getUri() {
        return $this->uri;
    }
    
    /**
     * @return string
     * @access public
     */
    public function getPath() {
        return $this->path;
    }

    /**
     * @param string $key
     * @access public
     */
    public function getParameter( string $key, $default = null ) {
        return $this->request->get( $key, $default );
    }

    /**
     * @param string $key
     * @return boolean
     * @access public
     */
    public function hasParameter( string $key ) {
        return $this->request->has( $key );
    }
    
    /**
     * @return FilteredMap
     * @access public
     */
    public function getParameters() {
        return $this->request;
    }
    
    /**
     * @return FilteredMap
     * @access public
     */
    public function getCookies() {
        return $this->cookies;
    } 


    /**
     * Set single derived request attribute
     * @param string $key the attribute key
     * @param mixed $value the attribute value
     * @access public
     */
    public function setAttribute( string $key, $value ) {
        $this->attributes[$key] = $value;
    }

    /**
     * Retrieve a single derived request attribute
     * @param string $key the attribute key
     * @access public
     */
    public function getAttribute( string $key, $default = null ) {
        return ( isset( $this->attributes[$key] ) ) ? $this->attributes[$key] : $default;
    }

    /**
     * Retrieve attributes derived from the request
     * @return array Attributes derived from the request
     * @access public
     */
    public function getAttributes() {
        return $this->attributes;
    }

}

