<?php
/**
 * FilteredMap
 * @author Jérôme Brilland
 * @version 1.0
 * @copyright (c) 2022, Jérôme Brilland
 * @license http://www.gnu.org/licenses/gpl.txt GNU General Public License
 */
class FilteredMap {
    
    private $map;
    
    public function __construct( array $baseMap ) {
        $this->map = $baseMap;
    }
    
    public function has( string $name ) {
        return isset( $this->map[$name] );
    }
    
    public function get( string $name, $default = null ) {
        return $this->map[$name] ?? $default;
    }
    
    /**
     * @param string $key
     * @return integer
     * @access public
     */
    public function getInt( string $key, $default = null ) {
        return filter_var( $this->get( $key, $default ), FILTER_SANITIZE_NUMBER_INT );
    }

    /**
     * @param string $key
     * @return float
     * @access public
     */
    public function getFloat( string $key, $default = null ) {
        return filter_var( $this->get( $key, $default ), FILTER_SANITIZE_NUMBER_FLOAT );
    }

    /**
     * @param string $key
     * @return string
     * @access public
     */
    public function getAlpha( string $key, $default = null ) {
        return preg_replace( '/[^[:alpha:]]/', '', $this->get( $key, $default ) );
    }

    /**
     * @param string $key
     * @return string
     * @access public
     */
    public function getAlnum( string $key, $default = null ) {
        return preg_replace( '/[^[:alnum:]]/', '', $this->get( $key, $default ) );
    }

    /**
     * @param string $key
     * @return string
     * @access public
     */
    public function getString( string $key, $default = null, $filter = 'unsafe_raw', $options = 0 ) {
        return filter_var( (string)$this->get( $key, $default ), filter_id( $filter ), $options );
    }

    /**
     * @param string $key
     * @return string
     * @access public
     */
    public function getWord( string $key, $default = null ) {
        return preg_replace( '/[\W]/', '', $this->get( $key, $default ) );
    }

    /**
     * @param string $key
     * @return string
     * @access public
     */
    public function getURL( string $key, $default = null ) {
        return filter_var( $this->get( $key, $default ), FILTER_SANITIZE_URL );
    }

    /**
     * @param string $key
     * @return boolean
     * @access public
     */
    public function getBoolean( string $key, $default = false ) {
        return (boolean) $this->get( $key, $default );
    }

    /**
     * @param string $key
     * @return array
     * @access public
     */
    public function getArray( string $key, $default = null ) {
        return ( is_array( $this->get( $key, $default ) ) ) ? $this->get( $key, $default ) : null;
    }
}
