<?php
/**
 * Configuration class
 * @author Jérôme Brilland
 * @version 1.0
 * @copyright (c) 2019, Jérôme Brilland
 * @license http://www.gnu.org/licenses/gpl.txt GNU General Public License
 */
class Configuration implements ArrayAccess, IteratorAggregate, Countable {

    /**
     * @access private
     */
    private $configurationHandler;

    /**
     * @param IConfigurationHandler configurationHandler
     * @access public
     */
    public function __construct( IConfigurationHandler $configurationHandler ) {
        $this->configurationHandler = $configurationHandler->load();
    }

    /**
     * @param string key
     * @return mixed
     * @access public
     */
    public function get( $key, $default = null ) {
        return $this->configurationHandler->get( $key, $default );
    }

    /**
     * Whether or not an offset exists.
     * @param string offset An offset to check for.
     * @return boolean True if the offset exists, false otherwise.
     * @access public
     */
    public function offsetExists( $offset ) {
        return $this->configurationHandler->has( $offset );
    }

    /**
     * Returns the value at specified offset.
     * @param string offset The offset to retrieve.
     * @return string The value at specified offset.
     * @access public
     */
    public function offsetGet( $offset ) {
        if ( $this->configurationHandler->has( $offset ) ) {
            return $this->configurationHandler->get( $offset );
        } else {
            throw new OutOfBoundsException( 'Key ' . $offset . ' does not exist.' );
        }
    }

    /**
     * Assigns a value to the specified offset.
     * @param string offset The offset to assign the value to.
     * @param string value The value to set.
     * @access public
     */
    public function offsetSet( $offset, $value ) {
        $this->configurationHandler->set( $offset, $value );
    }

    /**
     * Unsets an offset.
     * @param string offset The offset to unset.
     * @access public
     */
    public function offsetUnset( $offset ) {
        $this->configurationHandler->remove( $offset );
    }

    /**
     * @access public
     */
    public function getIterator() {
        return $this->configurationHandler->getIterator();
    }

    /**
     * @access public
     */
    public function count() {
        return count( $this->configurationHandler->getIterator() );
    }
}
