<?php
namespace install\services;

use models\IUser;
use models\User;
use models\Role;
use models\UserRole;
use models\Permission;
use models\RolePermission;
use models\Template;
use models\Page;

/**
 * AddSQLData service
 * @author Jérôme Brilland
 * @version 2.0
 * @copyright (c) 2019-2021, Jérôme Brilland
 * @license http://www.gnu.org/licenses/gpl.txt GNU General Public License
 */
class AddSQLData {

    private $settings;

    public function __construct( array $settings ) {
        $this->settings = $settings;
    }

    public function perform( $listener ) {
        //settings
        $configurationHandler = \IoC::resolve( 'databaseConfigurationHandler' );
        $settings = new \Configuration( $configurationHandler );
        $settings['site_title'] = $this->settings['site_title'];
        $settings['site_description'] = $this->settings['site_description'];
        $settings['from_name'] = $this->settings['from_name'];
        $settings['from_email'] = $this->settings['from_email'];
        $settings['administration_theme'] = 'Lemon';
        $settings['plugins'] = serialize( [] );
        $settings['home_page'] = 1;
        try {
            $configurationHandler->save();
        } catch ( \Exception $e ) {
            $listener->installFailed( $e->getMessage() );
            return;
        }

        //administrator
        $role = new Role();
        $role->name = 'administrator';
        if ( !$role->save() ) {
            $listener->installFailed( array_shift( $role->getErrors() )[0] );
            return;
        }

        $permissions = [
            'view_users',
            'create_user',
            'update_user',
            'delete_user',
            'view_pages',
            'create_page',
            'update_page',
            'delete_page',
            'view_templates',
            'create_template',
            'update_template',
            'delete_template',
            'view_snippets',
            'create_snippet',
            'update_snippet',
            'delete_snippet',
            'manage_settings',
            'manage_plugins',
            'assign_user_roles'
        ];

        foreach ( $permissions as $name ) {
            $permission = new Permission( ['name' => $name] );
            $permission->save();
            $rolePermission = new RolePermission( ['role_id' => $role->role_id, 'permission_id' => $permission->permission_id] );
            $rolePermission->save();
        }

        $user = new User();
        $user->username = $this->settings['administrator_username'];
        $user->password = $this->settings['administrator_password'];
        $user->password_confirmation = $this->settings['administrator_password'];
        $user->email = $this->settings['administrator_email'];
        $user->status = User::ACTIVE;
        if ( !$user->save() ) {
            $listener->installFailed( array_shift( $user->getErrors() )[0] );
            return;
        }

        $userRole = new UserRole( ['user_id' => $user->user_id, 'role_id' => $role->role_id] );
        $userRole->save();

        //editor
        $role = new Role();
        $role->name = 'editor';
        if ( !$role->save() ) {
            $listener->installFailed( array_shift( $role->getErrors() )[0] );
            return;
        }

        $editorPermissions = [
            'view_pages',
            'create_page',
            'update_page',
            'delete_page',
            'view_templates',
            'create_template',
            'update_template',
            'delete_template',
            'view_snippets',
            'create_snippet',
            'update_snippet',
            'delete_snippet'
        ];

        $permissions = Permission::findMany( ['permission_id'], ['name' => $editorPermissions, 'ORDER' => ['permission_id' => 'ASC']] );

        foreach ( $permissions as $permission ) {
            $rolePermission = new RolePermission( ['role_id' => $role->role_id, 'permission_id' => $permission->permission_id] );
            $rolePermission->save();
        }

        $role = new Role();
        $role->name = 'developer';
        if ( !$role->save() ) {
            $listener->installFailed( array_shift( $role->getErrors() )[0] );
            return;
        }

        //layout
        $template = new Template();
        $template->name = '@layout';
        $template->content_type = 'text/html';
        $template->content = file_get_contents ( ROOT_DIR . 'install' . DS . 'templates' . DS . '@layout.html' );
        $template->save();

        //default page template
        $template = new Template();
        $template->name = 'Default';
        $template->content_type = 'text/html';
        $template->content = file_get_contents( ROOT_DIR . 'install' . DS . 'templates' . DS . 'default.html' );
        $template->save();

        //home page
        $page = new Page();
        $page->title = 'Home page';
        $page->language = 'en';
        $page->slug = 'home';
        $page->content = <<<'TEXY'
======= Welcome to your site =======
This is your home page.
TEXY;
        $page->template_id = $template->template_id;
        $page->status = Page::PUBLISHED;
        $page->publication_date = date( 'Y-m-d' );
        $page->publication_time = date( 'H:i:s' );
        $page->position = 1;
        $page->save();
    }

}
