<?php
namespace controllers;

use models\Page;
/**
 * Page controller
 * @author Jérôme Brilland
 * @version 3.0
 * @copyright (c) 2018-2021, Jérôme Brilland
 * @license http://www.gnu.org/licenses/gpl.txt GNU General Public License
 */
class PageController extends \CRUDController {

    protected $modelClassName = 'models\Page';

    public function doList() {
        $this->columns = [
            'page_id',
            'title',
            'status',
            'created_at',
            'updated_at'
        ];
        parent::doList();
        $authorization = \IoC::resolve( 'authorization' );
        $authorization->authorize( 'view_pages' );
        $i18n = \IoC::resolve( 'i18n' );
        $this->title = $i18n->__( 'pages' );
        $this->pages = $this->models;
    }

    public function doSearch() {
        $keywords = strtolower( $this->request->getParameters()->getString( 'keywords' ) );
        $this->where = ['OR' => ['title[~]' => $keywords, 'description[~]' => $keywords, 'content[~]' => $keywords]];
        $this->doList();
        $this->render( 'list.html' );
    }

    public function doAddform() {
        parent::doAddform();
        $authorization = \IoC::resolve( 'authorization' );
        $authorization->authorize( 'create_page' );
        $i18n = \IoC::resolve( 'i18n' );
        $this->title = $i18n->__( 'add_page' );
        $this->page->parent_id = $this->request->getAttribute( 'id', null );
    }

    public function doAdd() {
        $authorization = \IoC::resolve( 'authorization' );
        $authorization->authorize( 'create_page' );
        parent::doAdd();
    }

    public function doUpdateform() {
        parent::doUpdateform();
        $authorization = \IoC::resolve( 'authorization' );
        $authorization->authorize( 'update_page' );
        $i18n = \IoC::resolve( 'i18n' );
        $this->title = $i18n->__( 'update_page' );
    }

    public function doUpdate() {
        $authorization = \IoC::resolve( 'authorization' );
        $authorization->authorize( 'update_page' );
        parent::doUpdate();
    }

    protected function afterAdd() {
        if ( $this->request->getParameters()->getBoolean( 'continue', false ) ) {
            $this->redirect( BASE_URL . 'page/update/' . $this->model->page_id );
        } else {
            $this->redirect( BASE_URL . 'page/list' );
        }
    }

    protected function afterUpdate() {
        if ( $this->request->getParameters()->getBoolean( 'continue', false ) ) {
            $this->redirect( BASE_URL . 'page/update/' . $this->model->page_id );
        } else {
            $this->redirect( BASE_URL . 'page/list' );
        }
    }

    public function doDeleteForm() {
        $authorization = \IoC::resolve( 'authorization' );
        $authorization->authorize( 'delete_page' );
        parent::doDeleteForm();
    }

    public function doDelete() {
        $authorization = \IoC::resolve( 'authorization' );
        $authorization->authorize( 'delete_page' );
        parent::doDelete();
    }

    public function doShow() {
        \IoC::resolve( 'eventmediator' )->trigger( 'before_find_page', [$this] );
        
        $this->page = Page::findPublishedPage( $this->request->getAttribute( 'slug', null ) ?? (int)$this->request->getAttribute( 'id' ) );

        if ( empty( $this->page ) ) {
            \IoC::resolve( 'eventmediator' )->trigger( 'page_not_found', [$this, $this->request, $this->response] );
            $this->forward( 'controllers\\ErrorController', '404' );
        }

        $latte = \IoC::resolve( 'view' )->getLatte();
        $latte->getLoader()->addLoader( \IoC::resolve( 'latteDatabaseLoader' ) );
        $this->render( $this->page->template_name );

    }

}
