/**************************************************************************/
/*  core_bind.h                                                           */
/**************************************************************************/
/*                         This file is part of:                          */
/*                             GODOT ENGINE                               */
/*                        https://godotengine.org                         */
/**************************************************************************/
/* Copyright (c) 2014-present Godot Engine contributors (see AUTHORS.md). */
/* Copyright (c) 2007-2014 Juan Linietsky, Ariel Manzur.                  */
/*                                                                        */
/* Permission is hereby granted, free of charge, to any person obtaining  */
/* a copy of this software and associated documentation files (the        */
/* "Software"), to deal in the Software without restriction, including    */
/* without limitation the rights to use, copy, modify, merge, publish,    */
/* distribute, sublicense, and/or sell copies of the Software, and to     */
/* permit persons to whom the Software is furnished to do so, subject to  */
/* the following conditions:                                              */
/*                                                                        */
/* The above copyright notice and this permission notice shall be         */
/* included in all copies or substantial portions of the Software.        */
/*                                                                        */
/* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,        */
/* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF     */
/* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. */
/* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY   */
/* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,   */
/* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE      */
/* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                 */
/**************************************************************************/

#ifndef CORE_BIND_H
#define CORE_BIND_H

#include "core/debugger/engine_profiler.h"
#include "core/io/image.h"
#include "core/io/resource_loader.h"
#include "core/io/resource_saver.h"
#include "core/object/script_language.h"
#include "core/os/os.h"
#include "core/os/semaphore.h"
#include "core/os/thread.h"
#include "core/templates/safe_refcount.h"

class MainLoop;
template <typename T>
class TypedArray;

namespace core_bind {

class ResourceLoader : public Object {
	GDCLASS(ResourceLoader, Object);

protected:
	static void _bind_methods();
	static ResourceLoader *singleton;

public:
	enum ThreadLoadStatus {
		THREAD_LOAD_INVALID_RESOURCE,
		THREAD_LOAD_IN_PROGRESS,
		THREAD_LOAD_FAILED,
		THREAD_LOAD_LOADED
	};

	enum CacheMode {
		CACHE_MODE_IGNORE, // Resource and subresources do not use path cache, no path is set into resource.
		CACHE_MODE_REUSE, // Resource and subresources use patch cache, reuse existing loaded resources instead of loading from disk when available.
		CACHE_MODE_REPLACE, // Resource and subresource use path cache, but replace existing loaded resources when available with information from disk.
	};

	static ResourceLoader *get_singleton() { return singleton; }

	Error load_threaded_request(const String &p_path, const String &p_type_hint = "", bool p_use_sub_threads = false, CacheMode p_cache_mode = CACHE_MODE_REUSE);
	ThreadLoadStatus load_threaded_get_status(const String &p_path, Array r_progress = Array());
	Ref<Resource> load_threaded_get(const String &p_path);

	Ref<Resource> load(const String &p_path, const String &p_type_hint = "", CacheMode p_cache_mode = CACHE_MODE_REUSE);
	Vector<String> get_recognized_extensions_for_type(const String &p_type);
	void add_resource_format_loader(Ref<ResourceFormatLoader> p_format_loader, bool p_at_front);
	void remove_resource_format_loader(Ref<ResourceFormatLoader> p_format_loader);
	void set_abort_on_missing_resources(bool p_abort);
	PackedStringArray get_dependencies(const String &p_path);
	bool has_cached(const String &p_path);
	bool exists(const String &p_path, const String &p_type_hint = "");
	ResourceUID::ID get_resource_uid(const String &p_path);

	ResourceLoader() { singleton = this; }
};

class ResourceSaver : public Object {
	GDCLASS(ResourceSaver, Object);

protected:
	static void _bind_methods();
	static ResourceSaver *singleton;

public:
	enum SaverFlags {
		FLAG_NONE = 0,
		FLAG_RELATIVE_PATHS = 1,
		FLAG_BUNDLE_RESOURCES = 2,
		FLAG_CHANGE_PATH = 4,
		FLAG_OMIT_EDITOR_PROPERTIES = 8,
		FLAG_SAVE_BIG_ENDIAN = 16,
		FLAG_COMPRESS = 32,
		FLAG_REPLACE_SUBRESOURCE_PATHS = 64,
	};

	static ResourceSaver *get_singleton() { return singleton; }

	Error save(const Ref<Resource> &p_resource, const String &p_path, BitField<SaverFlags> p_flags);
	Vector<String> get_recognized_extensions(const Ref<Resource> &p_resource);
	void add_resource_format_saver(Ref<ResourceFormatSaver> p_format_saver, bool p_at_front);
	void remove_resource_format_saver(Ref<ResourceFormatSaver> p_format_saver);

	ResourceSaver() { singleton = this; }
};

class OS : public Object {
	GDCLASS(OS, Object);

	mutable HashMap<String, bool> feature_cache;

protected:
	static void _bind_methods();
	static OS *singleton;

public:
	enum RenderingDriver {
		RENDERING_DRIVER_VULKAN,
		RENDERING_DRIVER_OPENGL3,
	};

	virtual PackedStringArray get_connected_midi_inputs();
	virtual void open_midi_inputs();
	virtual void close_midi_inputs();

	void set_low_processor_usage_mode(bool p_enabled);
	bool is_in_low_processor_usage_mode() const;

	void set_low_processor_usage_mode_sleep_usec(int p_usec);
	int get_low_processor_usage_mode_sleep_usec() const;

	void set_delta_smoothing(bool p_enabled);
	bool is_delta_smoothing_enabled() const;

	void alert(const String &p_alert, const String &p_title = "ALERT!");
	void crash(const String &p_message);

	Vector<String> get_system_fonts() const;
	String get_system_font_path(const String &p_font_name, int p_weight = 400, int p_stretch = 100, bool p_italic = false) const;
	Vector<String> get_system_font_path_for_text(const String &p_font_name, const String &p_text, const String &p_locale = String(), const String &p_script = String(), int p_weight = 400, int p_stretch = 100, bool p_italic = false) const;
	String get_executable_path() const;
	String read_string_from_stdin();
	int execute(const String &p_path, const Vector<String> &p_arguments, Array r_output = Array(), bool p_read_stderr = false, bool p_open_console = false);
	int create_process(const String &p_path, const Vector<String> &p_arguments, bool p_open_console = false);
	int create_instance(const Vector<String> &p_arguments);
	Error kill(int p_pid);
	Error shell_open(String p_uri);
	Error shell_show_in_file_manager(String p_path, bool p_open_folder = true);

	bool is_process_running(int p_pid) const;
	int get_process_id() const;

	void set_restart_on_exit(bool p_restart, const Vector<String> &p_restart_arguments = Vector<String>());
	bool is_restart_on_exit_set() const;
	Vector<String> get_restart_on_exit_arguments() const;

	bool has_environment(const String &p_var) const;
	String get_environment(const String &p_var) const;
	void set_environment(const String &p_var, const String &p_value) const;
	void unset_environment(const String &p_var) const;

	String get_name() const;
	String get_distribution_name() const;
	String get_version() const;
	Vector<String> get_cmdline_args();
	Vector<String> get_cmdline_user_args();

	Vector<String> get_video_adapter_driver_info() const;

	String get_locale() const;
	String get_locale_language() const;

	String get_model_name() const;

	bool is_debug_build() const;

	String get_unique_id() const;

	String get_keycode_string(Key p_code) const;
	bool is_keycode_unicode(char32_t p_unicode) const;
	Key find_keycode_from_string(const String &p_code) const;

	void set_use_file_access_save_and_swap(bool p_enable);

	uint64_t get_static_memory_usage() const;
	uint64_t get_static_memory_peak_usage() const;
	Dictionary get_memory_info() const;

	void delay_usec(int p_usec) const;
	void delay_msec(int p_msec) const;
	uint64_t get_ticks_msec() const;
	uint64_t get_ticks_usec() const;

	bool is_userfs_persistent() const;

	bool is_stdout_verbose() const;

	int get_processor_count() const;
	String get_processor_name() const;

	enum SystemDir {
		SYSTEM_DIR_DESKTOP,
		SYSTEM_DIR_DCIM,
		SYSTEM_DIR_DOCUMENTS,
		SYSTEM_DIR_DOWNLOADS,
		SYSTEM_DIR_MOVIES,
		SYSTEM_DIR_MUSIC,
		SYSTEM_DIR_PICTURES,
		SYSTEM_DIR_RINGTONES,
	};

	String get_system_dir(SystemDir p_dir, bool p_shared_storage = true) const;

	Error move_to_trash(const String &p_path) const;
	String get_user_data_dir() const;
	String get_config_dir() const;
	String get_data_dir() const;
	String get_cache_dir() const;

	Error set_thread_name(const String &p_name);
	Thread::ID get_thread_caller_id() const;
	Thread::ID get_main_thread_id() const;

	bool has_feature(const String &p_feature) const;

	bool request_permission(const String &p_name);
	bool request_permissions();
	Vector<String> get_granted_permissions() const;

	static OS *get_singleton() { return singleton; }

	OS() { singleton = this; }
};

class Geometry2D : public Object {
	GDCLASS(Geometry2D, Object);

	static Geometry2D *singleton;

protected:
	static void _bind_methods();

public:
	static Geometry2D *get_singleton();
	Variant segment_intersects_segment(const Vector2 &p_from_a, const Vector2 &p_to_a, const Vector2 &p_from_b, const Vector2 &p_to_b);
	Variant line_intersects_line(const Vector2 &p_from_a, const Vector2 &p_dir_a, const Vector2 &p_from_b, const Vector2 &p_dir_b);
	Vector<Vector2> get_closest_points_between_segments(const Vector2 &p1, const Vector2 &q1, const Vector2 &p2, const Vector2 &q2);
	Vector2 get_closest_point_to_segment(const Vector2 &p_point, const Vector2 &p_a, const Vector2 &p_b);
	Vector2 get_closest_point_to_segment_uncapped(const Vector2 &p_point, const Vector2 &p_a, const Vector2 &p_b);
	bool point_is_inside_triangle(const Vector2 &s, const Vector2 &a, const Vector2 &b, const Vector2 &c) const;

	bool is_point_in_circle(const Vector2 &p_point, const Vector2 &p_circle_pos, real_t p_circle_radius);
	real_t segment_intersects_circle(const Vector2 &p_from, const Vector2 &p_to, const Vector2 &p_circle_pos, real_t p_circle_radius);

	bool is_polygon_clockwise(const Vector<Vector2> &p_polygon);
	bool is_point_in_polygon(const Point2 &p_point, const Vector<Vector2> &p_polygon);
	Vector<int> triangulate_polygon(const Vector<Vector2> &p_polygon);
	Vector<int> triangulate_delaunay(const Vector<Vector2> &p_points);
	Vector<Point2> convex_hull(const Vector<Point2> &p_points);
	TypedArray<PackedVector2Array> decompose_polygon_in_convex(const Vector<Vector2> &p_polygon);

	enum PolyBooleanOperation {
		OPERATION_UNION,
		OPERATION_DIFFERENCE,
		OPERATION_INTERSECTION,
		OPERATION_XOR
	};
	// 2D polygon boolean operations.
	TypedArray<PackedVector2Array> merge_polygons(const Vector<Vector2> &p_polygon_a, const Vector<Vector2> &p_polygon_b); // Union (add).
	TypedArray<PackedVector2Array> clip_polygons(const Vector<Vector2> &p_polygon_a, const Vector<Vector2> &p_polygon_b); // Difference (subtract).
	TypedArray<PackedVector2Array> intersect_polygons(const Vector<Vector2> &p_polygon_a, const Vector<Vector2> &p_polygon_b); // Common area (multiply).
	TypedArray<PackedVector2Array> exclude_polygons(const Vector<Vector2> &p_polygon_a, const Vector<Vector2> &p_polygon_b); // All but common area (xor).

	// 2D polyline vs polygon operations.
	TypedArray<PackedVector2Array> clip_polyline_with_polygon(const Vector<Vector2> &p_polyline, const Vector<Vector2> &p_polygon); // Cut.
	TypedArray<PackedVector2Array> intersect_polyline_with_polygon(const Vector<Vector2> &p_polyline, const Vector<Vector2> &p_polygon); // Chop.

	// 2D offset polygons/polylines.
	enum PolyJoinType {
		JOIN_SQUARE,
		JOIN_ROUND,
		JOIN_MITER
	};
	enum PolyEndType {
		END_POLYGON,
		END_JOINED,
		END_BUTT,
		END_SQUARE,
		END_ROUND
	};
	TypedArray<PackedVector2Array> offset_polygon(const Vector<Vector2> &p_polygon, real_t p_delta, PolyJoinType p_join_type = JOIN_SQUARE);
	TypedArray<PackedVector2Array> offset_polyline(const Vector<Vector2> &p_polygon, real_t p_delta, PolyJoinType p_join_type = JOIN_SQUARE, PolyEndType p_end_type = END_SQUARE);

	Dictionary make_atlas(const Vector<Size2> &p_rects);

	Geometry2D() { singleton = this; }
};

class Geometry3D : public Object {
	GDCLASS(Geometry3D, Object);

	static Geometry3D *singleton;

protected:
	static void _bind_methods();

public:
	static Geometry3D *get_singleton();
	TypedArray<Plane> build_box_planes(const Vector3 &p_extents);
	TypedArray<Plane> build_cylinder_planes(float p_radius, float p_height, int p_sides, Vector3::Axis p_axis = Vector3::AXIS_Z);
	TypedArray<Plane> build_capsule_planes(float p_radius, float p_height, int p_sides, int p_lats, Vector3::Axis p_axis = Vector3::AXIS_Z);
	Vector<Vector3> get_closest_points_between_segments(const Vector3 &p1, const Vector3 &p2, const Vector3 &q1, const Vector3 &q2);
	Vector3 get_closest_point_to_segment(const Vector3 &p_point, const Vector3 &p_a, const Vector3 &p_b);
	Vector3 get_closest_point_to_segment_uncapped(const Vector3 &p_point, const Vector3 &p_a, const Vector3 &p_b);
	Variant ray_intersects_triangle(const Vector3 &p_from, const Vector3 &p_dir, const Vector3 &p_v0, const Vector3 &p_v1, const Vector3 &p_v2);
	Variant segment_intersects_triangle(const Vector3 &p_from, const Vector3 &p_to, const Vector3 &p_v0, const Vector3 &p_v1, const Vector3 &p_v2);

	Vector<Vector3> segment_intersects_sphere(const Vector3 &p_from, const Vector3 &p_to, const Vector3 &p_sphere_pos, real_t p_sphere_radius);
	Vector<Vector3> segment_intersects_cylinder(const Vector3 &p_from, const Vector3 &p_to, float p_height, float p_radius);
	Vector<Vector3> segment_intersects_convex(const Vector3 &p_from, const Vector3 &p_to, const TypedArray<Plane> &p_planes);

	Vector<Vector3> clip_polygon(const Vector<Vector3> &p_points, const Plane &p_plane);

	Geometry3D() { singleton = this; }
};

class Marshalls : public Object {
	GDCLASS(Marshalls, Object);

	static Marshalls *singleton;

protected:
	static void _bind_methods();

public:
	static Marshalls *get_singleton();

	String variant_to_base64(const Variant &p_var, bool p_full_objects = false);
	Variant base64_to_variant(const String &p_str, bool p_allow_objects = false);

	String raw_to_base64(const Vector<uint8_t> &p_arr);
	Vector<uint8_t> base64_to_raw(const String &p_str);

	String utf8_to_base64(const String &p_str);
	String base64_to_utf8(const String &p_str);

	Marshalls() { singleton = this; }
	~Marshalls() { singleton = nullptr; }
};

class Mutex : public RefCounted {
	GDCLASS(Mutex, RefCounted);
	::Mutex mutex;

	static void _bind_methods();

public:
	void lock();
	bool try_lock();
	void unlock();
};

class Semaphore : public RefCounted {
	GDCLASS(Semaphore, RefCounted);
	::Semaphore semaphore;

	static void _bind_methods();

public:
	void wait();
	bool try_wait();
	void post();
};

class Thread : public RefCounted {
	GDCLASS(Thread, RefCounted);

protected:
	Variant ret;
	SafeFlag running;
	Callable target_callable;
	::Thread thread;
	static void _bind_methods();
	static void _start_func(void *ud);

public:
	enum Priority {
		PRIORITY_LOW,
		PRIORITY_NORMAL,
		PRIORITY_HIGH,
		PRIORITY_MAX
	};

	Error start(const Callable &p_callable, Priority p_priority = PRIORITY_NORMAL);
	String get_id() const;
	bool is_started() const;
	bool is_alive() const;
	Variant wait_to_finish();
};

namespace special {

class ClassDB : public Object {
	GDCLASS(ClassDB, Object);

protected:
	static void _bind_methods();

public:
	PackedStringArray get_class_list() const;
	PackedStringArray get_inheriters_from_class(const StringName &p_class) const;
	StringName get_parent_class(const StringName &p_class) const;
	bool class_exists(const StringName &p_class) const;
	bool is_parent_class(const StringName &p_class, const StringName &p_inherits) const;
	bool can_instantiate(const StringName &p_class) const;
	Variant instantiate(const StringName &p_class) const;

	bool class_has_signal(StringName p_class, StringName p_signal) const;
	Dictionary class_get_signal(StringName p_class, StringName p_signal) const;
	TypedArray<Dictionary> class_get_signal_list(StringName p_class, bool p_no_inheritance = false) const;

	TypedArray<Dictionary> class_get_property_list(StringName p_class, bool p_no_inheritance = false) const;
	Variant class_get_property(Object *p_object, const StringName &p_property) const;
	Error class_set_property(Object *p_object, const StringName &p_property, const Variant &p_value) const;

	bool class_has_method(StringName p_class, StringName p_method, bool p_no_inheritance = false) const;

	TypedArray<Dictionary> class_get_method_list(StringName p_class, bool p_no_inheritance = false) const;

	PackedStringArray class_get_integer_constant_list(const StringName &p_class, bool p_no_inheritance = false) const;
	bool class_has_integer_constant(const StringName &p_class, const StringName &p_name) const;
	int64_t class_get_integer_constant(const StringName &p_class, const StringName &p_name) const;

	bool class_has_enum(const StringName &p_class, const StringName &p_name, bool p_no_inheritance = false) const;
	PackedStringArray class_get_enum_list(const StringName &p_class, bool p_no_inheritance = false) const;
	PackedStringArray class_get_enum_constants(const StringName &p_class, const StringName &p_enum, bool p_no_inheritance = false) const;
	StringName class_get_integer_constant_enum(const StringName &p_class, const StringName &p_name, bool p_no_inheritance = false) const;

	bool is_class_enabled(StringName p_class) const;

	ClassDB() {}
	~ClassDB() {}
};

} // namespace special

class Engine : public Object {
	GDCLASS(Engine, Object);

protected:
	static void _bind_methods();
	static Engine *singleton;

public:
	static Engine *get_singleton() { return singleton; }
	void set_physics_ticks_per_second(int p_ips);
	int get_physics_ticks_per_second() const;

	void set_max_physics_steps_per_frame(int p_max_physics_steps);
	int get_max_physics_steps_per_frame() const;

	void set_physics_jitter_fix(double p_threshold);
	double get_physics_jitter_fix() const;
	double get_physics_interpolation_fraction() const;

	void set_max_fps(int p_fps);
	int get_max_fps() const;

	double get_frames_per_second() const;
	uint64_t get_physics_frames() const;
	uint64_t get_process_frames() const;

	int get_frames_drawn();

	void set_time_scale(double p_scale);
	double get_time_scale();

	MainLoop *get_main_loop() const;

	Dictionary get_version_info() const;
	Dictionary get_author_info() const;
	TypedArray<Dictionary> get_copyright_info() const;
	Dictionary get_donor_info() const;
	Dictionary get_license_info() const;
	String get_license_text() const;

	String get_architecture_name() const;

	bool is_in_physics_frame() const;

	bool has_singleton(const StringName &p_name) const;
	Object *get_singleton_object(const StringName &p_name) const;
	void register_singleton(const StringName &p_name, Object *p_object);
	void unregister_singleton(const StringName &p_name);
	Vector<String> get_singleton_list() const;

	Error register_script_language(ScriptLanguage *p_language);
	Error unregister_script_language(const ScriptLanguage *p_language);
	int get_script_language_count();
	ScriptLanguage *get_script_language(int p_index) const;

	void set_editor_hint(bool p_enabled);
	bool is_editor_hint() const;

	// `set_write_movie_path()` is not exposed to the scripting API as changing it at run-time has no effect.
	String get_write_movie_path() const;

	void set_print_error_messages(bool p_enabled);
	bool is_printing_error_messages() const;

	Engine() { singleton = this; }
};

class EngineDebugger : public Object {
	GDCLASS(EngineDebugger, Object);

	HashMap<StringName, Callable> captures;
	HashMap<StringName, Ref<EngineProfiler>> profilers;

protected:
	static void _bind_methods();
	static EngineDebugger *singleton;

public:
	static EngineDebugger *get_singleton() { return singleton; }

	bool is_active();

	void register_profiler(const StringName &p_name, Ref<EngineProfiler> p_profiler);
	void unregister_profiler(const StringName &p_name);
	bool is_profiling(const StringName &p_name);
	bool has_profiler(const StringName &p_name);
	void profiler_add_frame_data(const StringName &p_name, const Array &p_data);
	void profiler_enable(const StringName &p_name, bool p_enabled, const Array &p_opts = Array());

	void register_message_capture(const StringName &p_name, const Callable &p_callable);
	void unregister_message_capture(const StringName &p_name);
	bool has_capture(const StringName &p_name);

	void send_message(const String &p_msg, const Array &p_data);

	static Error call_capture(void *p_user, const String &p_cmd, const Array &p_data, bool &r_captured);

	EngineDebugger() { singleton = this; }
	~EngineDebugger();
};

} // namespace core_bind

VARIANT_ENUM_CAST(core_bind::ResourceLoader::ThreadLoadStatus);
VARIANT_ENUM_CAST(core_bind::ResourceLoader::CacheMode);

VARIANT_BITFIELD_CAST(core_bind::ResourceSaver::SaverFlags);

VARIANT_ENUM_CAST(core_bind::OS::RenderingDriver);
VARIANT_ENUM_CAST(core_bind::OS::SystemDir);

VARIANT_ENUM_CAST(core_bind::Geometry2D::PolyBooleanOperation);
VARIANT_ENUM_CAST(core_bind::Geometry2D::PolyJoinType);
VARIANT_ENUM_CAST(core_bind::Geometry2D::PolyEndType);

VARIANT_ENUM_CAST(core_bind::Thread::Priority);

#endif // CORE_BIND_H
