/**************************************************************************/
/*  action_map_editor.cpp                                                 */
/**************************************************************************/
/*                         This file is part of:                          */
/*                             GODOT ENGINE                               */
/*                        https://godotengine.org                         */
/**************************************************************************/
/* Copyright (c) 2014-present Godot Engine contributors (see AUTHORS.md). */
/* Copyright (c) 2007-2014 Juan Linietsky, Ariel Manzur.                  */
/*                                                                        */
/* Permission is hereby granted, free of charge, to any person obtaining  */
/* a copy of this software and associated documentation files (the        */
/* "Software"), to deal in the Software without restriction, including    */
/* without limitation the rights to use, copy, modify, merge, publish,    */
/* distribute, sublicense, and/or sell copies of the Software, and to     */
/* permit persons to whom the Software is furnished to do so, subject to  */
/* the following conditions:                                              */
/*                                                                        */
/* The above copyright notice and this permission notice shall be         */
/* included in all copies or substantial portions of the Software.        */
/*                                                                        */
/* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,        */
/* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF     */
/* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. */
/* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY   */
/* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,   */
/* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE      */
/* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                 */
/**************************************************************************/

#include "editor/action_map_editor.h"

#include "editor/editor_scale.h"
#include "editor/event_listener_line_edit.h"
#include "editor/input_event_configuration_dialog.h"
#include "scene/gui/check_button.h"
#include "scene/gui/tree.h"
#include "scene/scene_string_names.h"

static bool _is_action_name_valid(const String &p_name) {
	const char32_t *cstr = p_name.get_data();
	for (int i = 0; cstr[i]; i++) {
		if (cstr[i] == '/' || cstr[i] == ':' || cstr[i] == '"' ||
				cstr[i] == '=' || cstr[i] == '\\' || cstr[i] < 32) {
			return false;
		}
	}
	return true;
}

void ActionMapEditor::_event_config_confirmed() {
	Ref<InputEvent> ev = event_config_dialog->get_event();

	Dictionary new_action = current_action.duplicate();
	Array events = new_action["events"].duplicate();

	if (current_action_event_index == -1) {
		// Add new event
		events.push_back(ev);
	} else {
		// Edit existing event
		events[current_action_event_index] = ev;
	}

	new_action["events"] = events;
	emit_signal(SNAME("action_edited"), current_action_name, new_action);
}

void ActionMapEditor::_add_action_pressed() {
	_add_action(add_edit->get_text());
}

String ActionMapEditor::_check_new_action_name(const String &p_name) {
	if (p_name.is_empty() || !_is_action_name_valid(p_name)) {
		return TTR("Invalid action name. It cannot be empty nor contain '/', ':', '=', '\\' or '\"'");
	}

	if (_has_action(p_name)) {
		return vformat(TTR("An action with the name '%s' already exists."), p_name);
	}

	return "";
}

void ActionMapEditor::_add_edit_text_changed(const String &p_name) {
	String error = _check_new_action_name(p_name);
	add_button->set_tooltip_text(error);
	add_button->set_disabled(!error.is_empty());
}

bool ActionMapEditor::_has_action(const String &p_name) const {
	for (const ActionInfo &action_info : actions_cache) {
		if (p_name == action_info.name) {
			return true;
		}
	}
	return false;
}

void ActionMapEditor::_add_action(const String &p_name) {
	String error = _check_new_action_name(p_name);
	if (!error.is_empty()) {
		show_message(error);
		return;
	}

	add_edit->clear();
	emit_signal(SNAME("action_added"), p_name);
}

void ActionMapEditor::_action_edited() {
	TreeItem *ti = action_tree->get_edited();
	if (!ti) {
		return;
	}

	if (action_tree->get_selected_column() == 0) {
		// Name Edited
		String new_name = ti->get_text(0);
		String old_name = ti->get_meta("__name");

		if (new_name == old_name) {
			return;
		}

		if (new_name.is_empty() || !_is_action_name_valid(new_name)) {
			ti->set_text(0, old_name);
			show_message(TTR("Invalid action name. It cannot be empty nor contain '/', ':', '=', '\\' or '\"'"));
			return;
		}

		if (_has_action(new_name)) {
			ti->set_text(0, old_name);
			show_message(vformat(TTR("An action with the name '%s' already exists."), new_name));
			return;
		}

		emit_signal(SNAME("action_renamed"), old_name, new_name);
	} else if (action_tree->get_selected_column() == 1) {
		// Deadzone Edited
		String name = ti->get_meta("__name");
		Dictionary old_action = ti->get_meta("__action");
		Dictionary new_action = old_action.duplicate();
		new_action["deadzone"] = ti->get_range(1);

		// Call deferred so that input can finish propagating through tree, allowing re-making of tree to occur.
		call_deferred(SNAME("emit_signal"), "action_edited", name, new_action);
	}
}

void ActionMapEditor::_tree_button_pressed(Object *p_item, int p_column, int p_id, MouseButton p_button) {
	if (p_button != MouseButton::LEFT) {
		return;
	}

	ItemButton option = (ItemButton)p_id;

	TreeItem *item = Object::cast_to<TreeItem>(p_item);
	if (!item) {
		return;
	}

	switch (option) {
		case ActionMapEditor::BUTTON_ADD_EVENT: {
			current_action = item->get_meta("__action");
			current_action_name = item->get_meta("__name");
			current_action_event_index = -1;

			event_config_dialog->popup_and_configure();
		} break;
		case ActionMapEditor::BUTTON_EDIT_EVENT: {
			// Action and Action name is located on the parent of the event.
			current_action = item->get_parent()->get_meta("__action");
			current_action_name = item->get_parent()->get_meta("__name");

			current_action_event_index = item->get_meta("__index");

			Ref<InputEvent> ie = item->get_meta("__event");
			if (ie.is_valid()) {
				event_config_dialog->popup_and_configure(ie);
			}
		} break;
		case ActionMapEditor::BUTTON_REMOVE_ACTION: {
			// Send removed action name
			String name = item->get_meta("__name");
			emit_signal(SNAME("action_removed"), name);
		} break;
		case ActionMapEditor::BUTTON_REMOVE_EVENT: {
			// Remove event and send updated action
			Dictionary action = item->get_parent()->get_meta("__action").duplicate();
			String action_name = item->get_parent()->get_meta("__name");

			int event_index = item->get_meta("__index");

			Array events = action["events"].duplicate();
			events.remove_at(event_index);
			action["events"] = events;

			emit_signal(SNAME("action_edited"), action_name, action);
		} break;
		default:
			break;
	}
}

void ActionMapEditor::_tree_item_activated() {
	TreeItem *item = action_tree->get_selected();

	if (!item || !item->has_meta("__event")) {
		return;
	}

	_tree_button_pressed(item, 2, BUTTON_EDIT_EVENT, MouseButton::LEFT);
}

void ActionMapEditor::set_show_builtin_actions(bool p_show) {
	show_builtin_actions = p_show;
	show_builtin_actions_checkbutton->set_pressed(p_show);

	// Prevent unnecessary updates of action list when cache is empty.
	if (!actions_cache.is_empty()) {
		update_action_list();
	}
}

void ActionMapEditor::_search_term_updated(const String &) {
	update_action_list();
}

void ActionMapEditor::_search_by_event(const Ref<InputEvent> &p_event) {
	if (p_event.is_null() || (p_event->is_pressed() && !p_event->is_echo())) {
		update_action_list();
	}
}

Variant ActionMapEditor::get_drag_data_fw(const Point2 &p_point, Control *p_from) {
	TreeItem *selected = action_tree->get_selected();
	if (!selected) {
		return Variant();
	}

	String name = selected->get_text(0);
	Label *label = memnew(Label(name));
	label->set_theme_type_variation("HeaderSmall");
	label->set_modulate(Color(1, 1, 1, 1.0f));
	action_tree->set_drag_preview(label);

	Dictionary drag_data;

	if (selected->has_meta("__action")) {
		drag_data["input_type"] = "action";
	}

	if (selected->has_meta("__event")) {
		drag_data["input_type"] = "event";
	}

	action_tree->set_drop_mode_flags(Tree::DROP_MODE_INBETWEEN);

	return drag_data;
}

bool ActionMapEditor::can_drop_data_fw(const Point2 &p_point, const Variant &p_data, Control *p_from) const {
	Dictionary d = p_data;
	if (!d.has("input_type")) {
		return false;
	}

	TreeItem *selected = action_tree->get_selected();
	TreeItem *item = action_tree->get_item_at_position(p_point);
	if (!selected || !item || item == selected) {
		return false;
	}

	// Don't allow moving an action in-between events.
	if (d["input_type"] == "action" && item->has_meta("__event")) {
		return false;
	}

	// Don't allow moving an event to a different action.
	if (d["input_type"] == "event" && item->get_parent() != selected->get_parent()) {
		return false;
	}

	return true;
}

void ActionMapEditor::drop_data_fw(const Point2 &p_point, const Variant &p_data, Control *p_from) {
	if (!can_drop_data_fw(p_point, p_data, p_from)) {
		return;
	}

	TreeItem *selected = action_tree->get_selected();
	TreeItem *target = action_tree->get_item_at_position(p_point);
	bool drop_above = action_tree->get_drop_section_at_position(p_point) == -1;

	if (!target) {
		return;
	}

	Dictionary d = p_data;
	if (d["input_type"] == "action") {
		// Change action order.
		String relative_to = target->get_meta("__name");
		String action_name = selected->get_meta("__name");
		emit_signal(SNAME("action_reordered"), action_name, relative_to, drop_above);

	} else if (d["input_type"] == "event") {
		// Change event order
		int current_index = selected->get_meta("__index");
		int target_index = target->get_meta("__index");

		// Construct new events array.
		Dictionary new_action = selected->get_parent()->get_meta("__action");

		Array events = new_action["events"];
		Array new_events;

		// The following method was used to perform the array changes since `remove` followed by `insert` was not working properly at time of writing.
		// Loop thought existing events
		for (int i = 0; i < events.size(); i++) {
			// If you come across the current index, just skip it, as it has been moved.
			if (i == current_index) {
				continue;
			} else if (i == target_index) {
				// We are at the target index. If drop above, add selected event there first, then target, so moved event goes on top.
				if (drop_above) {
					new_events.push_back(events[current_index]);
					new_events.push_back(events[target_index]);
				} else {
					new_events.push_back(events[target_index]);
					new_events.push_back(events[current_index]);
				}
			} else {
				new_events.push_back(events[i]);
			}
		}

		new_action["events"] = new_events;
		emit_signal(SNAME("action_edited"), selected->get_parent()->get_meta("__name"), new_action);
	}
}

void ActionMapEditor::_notification(int p_what) {
	switch (p_what) {
		case NOTIFICATION_ENTER_TREE:
		case NOTIFICATION_THEME_CHANGED: {
			action_list_search->set_right_icon(get_theme_icon(SNAME("Search"), SNAME("EditorIcons")));
			if (!actions_cache.is_empty()) {
				update_action_list();
			}
		} break;
	}
}

void ActionMapEditor::_bind_methods() {
	ClassDB::bind_method(D_METHOD("_get_drag_data_fw"), &ActionMapEditor::get_drag_data_fw);
	ClassDB::bind_method(D_METHOD("_can_drop_data_fw"), &ActionMapEditor::can_drop_data_fw);
	ClassDB::bind_method(D_METHOD("_drop_data_fw"), &ActionMapEditor::drop_data_fw);

	ADD_SIGNAL(MethodInfo("action_added", PropertyInfo(Variant::STRING, "name")));
	ADD_SIGNAL(MethodInfo("action_edited", PropertyInfo(Variant::STRING, "name"), PropertyInfo(Variant::DICTIONARY, "new_action")));
	ADD_SIGNAL(MethodInfo("action_removed", PropertyInfo(Variant::STRING, "name")));
	ADD_SIGNAL(MethodInfo("action_renamed", PropertyInfo(Variant::STRING, "old_name"), PropertyInfo(Variant::STRING, "new_name")));
	ADD_SIGNAL(MethodInfo("action_reordered", PropertyInfo(Variant::STRING, "action_name"), PropertyInfo(Variant::STRING, "relative_to"), PropertyInfo(Variant::BOOL, "before")));
	ADD_SIGNAL(MethodInfo(SNAME("filter_focused")));
	ADD_SIGNAL(MethodInfo(SNAME("filter_unfocused")));
}

LineEdit *ActionMapEditor::get_search_box() const {
	return action_list_search;
}

InputEventConfigurationDialog *ActionMapEditor::get_configuration_dialog() {
	return event_config_dialog;
}

bool ActionMapEditor::_should_display_action(const String &p_name, const Array &p_events) const {
	const Ref<InputEvent> search_ev = action_list_search_by_event->get_event();
	bool event_match = true;
	if (search_ev.is_valid()) {
		event_match = false;
		for (int i = 0; i < p_events.size(); ++i) {
			const Ref<InputEvent> ev = p_events[i];
			if (ev.is_valid() && ev->is_match(search_ev, true)) {
				event_match = true;
			}
		}
	}

	return event_match && action_list_search->get_text().is_subsequence_ofn(p_name);
}

void ActionMapEditor::update_action_list(const Vector<ActionInfo> &p_action_infos) {
	if (!p_action_infos.is_empty()) {
		actions_cache = p_action_infos;
	}

	action_tree->clear();
	TreeItem *root = action_tree->create_item();

	for (int i = 0; i < actions_cache.size(); i++) {
		ActionInfo action_info = actions_cache[i];

		const Array events = action_info.action["events"];
		if (!_should_display_action(action_info.name, events)) {
			continue;
		}

		if (!action_info.editable && !show_builtin_actions) {
			continue;
		}

		const Variant deadzone = action_info.action["deadzone"];

		// Update Tree...

		TreeItem *action_item = action_tree->create_item(root);
		action_item->set_meta("__action", action_info.action);
		action_item->set_meta("__name", action_info.name);

		// First Column - Action Name
		action_item->set_text(0, action_info.name);
		action_item->set_editable(0, action_info.editable);
		action_item->set_icon(0, action_info.icon);

		// Second Column - Deadzone
		action_item->set_editable(1, true);
		action_item->set_cell_mode(1, TreeItem::CELL_MODE_RANGE);
		action_item->set_range_config(1, 0.0, 1.0, 0.01);
		action_item->set_range(1, deadzone);

		// Third column - buttons
		action_item->add_button(2, action_tree->get_theme_icon(SNAME("Add"), SNAME("EditorIcons")), BUTTON_ADD_EVENT, false, TTR("Add Event"));
		action_item->add_button(2, action_tree->get_theme_icon(SNAME("Remove"), SNAME("EditorIcons")), BUTTON_REMOVE_ACTION, !action_info.editable, action_info.editable ? TTR("Remove Action") : TTR("Cannot Remove Action"));

		action_item->set_custom_bg_color(0, action_tree->get_theme_color(SNAME("prop_subsection"), SNAME("Editor")));
		action_item->set_custom_bg_color(1, action_tree->get_theme_color(SNAME("prop_subsection"), SNAME("Editor")));

		for (int evnt_idx = 0; evnt_idx < events.size(); evnt_idx++) {
			Ref<InputEvent> event = events[evnt_idx];
			if (event.is_null()) {
				continue;
			}

			TreeItem *event_item = action_tree->create_item(action_item);

			// First Column - Text
			event_item->set_text(0, EventListenerLineEdit::get_event_text(event, true));
			event_item->set_meta("__event", event);
			event_item->set_meta("__index", evnt_idx);

			// First Column - Icon
			Ref<InputEventKey> k = event;
			if (k.is_valid()) {
				if (k->get_physical_keycode() == Key::NONE) {
					event_item->set_icon(0, action_tree->get_theme_icon(SNAME("Keyboard"), SNAME("EditorIcons")));
				} else {
					event_item->set_icon(0, action_tree->get_theme_icon(SNAME("KeyboardPhysical"), SNAME("EditorIcons")));
				}
			}

			Ref<InputEventMouseButton> mb = event;
			if (mb.is_valid()) {
				event_item->set_icon(0, action_tree->get_theme_icon(SNAME("Mouse"), SNAME("EditorIcons")));
			}

			Ref<InputEventJoypadButton> jb = event;
			if (jb.is_valid()) {
				event_item->set_icon(0, action_tree->get_theme_icon(SNAME("JoyButton"), SNAME("EditorIcons")));
			}

			Ref<InputEventJoypadMotion> jm = event;
			if (jm.is_valid()) {
				event_item->set_icon(0, action_tree->get_theme_icon(SNAME("JoyAxis"), SNAME("EditorIcons")));
			}

			// Third Column - Buttons
			event_item->add_button(2, action_tree->get_theme_icon(SNAME("Edit"), SNAME("EditorIcons")), BUTTON_EDIT_EVENT, false, TTR("Edit Event"));
			event_item->add_button(2, action_tree->get_theme_icon(SNAME("Remove"), SNAME("EditorIcons")), BUTTON_REMOVE_EVENT, false, TTR("Remove Event"));
			event_item->set_button_color(2, 0, Color(1, 1, 1, 0.75));
			event_item->set_button_color(2, 1, Color(1, 1, 1, 0.75));
		}
	}
}

void ActionMapEditor::show_message(const String &p_message) {
	message->set_text(p_message);
	message->popup_centered();
}

void ActionMapEditor::use_external_search_box(LineEdit *p_searchbox) {
	memdelete(action_list_search);
	action_list_search = p_searchbox;
	action_list_search->connect("text_changed", callable_mp(this, &ActionMapEditor::_search_term_updated));
}

void ActionMapEditor::_on_filter_focused() {
	emit_signal(SNAME("filter_focused"));
}

void ActionMapEditor::_on_filter_unfocused() {
	emit_signal(SNAME("filter_unfocused"));
}

ActionMapEditor::ActionMapEditor() {
	// Main Vbox Container
	VBoxContainer *main_vbox = memnew(VBoxContainer);
	main_vbox->set_anchors_and_offsets_preset(PRESET_FULL_RECT);
	add_child(main_vbox);

	HBoxContainer *top_hbox = memnew(HBoxContainer);
	main_vbox->add_child(top_hbox);

	action_list_search = memnew(LineEdit);
	action_list_search->set_h_size_flags(Control::SIZE_EXPAND_FILL);
	action_list_search->set_placeholder(TTR("Filter by name..."));
	action_list_search->set_clear_button_enabled(true);
	action_list_search->connect("text_changed", callable_mp(this, &ActionMapEditor::_search_term_updated));
	top_hbox->add_child(action_list_search);

	action_list_search_by_event = memnew(EventListenerLineEdit);
	action_list_search_by_event->set_h_size_flags(Control::SIZE_EXPAND_FILL);
	action_list_search_by_event->set_stretch_ratio(0.75);
	action_list_search_by_event->connect("event_changed", callable_mp(this, &ActionMapEditor::_search_by_event));
	action_list_search_by_event->connect(SceneStringNames::get_singleton()->focus_entered, callable_mp(this, &ActionMapEditor::_on_filter_focused));
	action_list_search_by_event->connect(SceneStringNames::get_singleton()->focus_exited, callable_mp(this, &ActionMapEditor::_on_filter_unfocused));
	top_hbox->add_child(action_list_search_by_event);

	Button *clear_all_search = memnew(Button);
	clear_all_search->set_text(TTR("Clear All"));
	clear_all_search->connect("pressed", callable_mp(action_list_search_by_event, &EventListenerLineEdit::clear_event));
	clear_all_search->connect("pressed", callable_mp(action_list_search, &LineEdit::clear));
	top_hbox->add_child(clear_all_search);

	// Adding Action line edit + button
	add_hbox = memnew(HBoxContainer);
	add_hbox->set_h_size_flags(Control::SIZE_EXPAND_FILL);

	add_edit = memnew(LineEdit);
	add_edit->set_h_size_flags(Control::SIZE_EXPAND_FILL);
	add_edit->set_placeholder(TTR("Add New Action"));
	add_edit->set_clear_button_enabled(true);
	add_edit->connect("text_changed", callable_mp(this, &ActionMapEditor::_add_edit_text_changed));
	add_edit->connect("text_submitted", callable_mp(this, &ActionMapEditor::_add_action));
	add_hbox->add_child(add_edit);

	add_button = memnew(Button);
	add_button->set_text(TTR("Add"));
	add_button->connect("pressed", callable_mp(this, &ActionMapEditor::_add_action_pressed));
	add_hbox->add_child(add_button);
	// Disable the button and set its tooltip.
	_add_edit_text_changed(add_edit->get_text());

	show_builtin_actions_checkbutton = memnew(CheckButton);
	show_builtin_actions_checkbutton->set_pressed(false);
	show_builtin_actions_checkbutton->set_text(TTR("Show Built-in Actions"));
	show_builtin_actions_checkbutton->connect("toggled", callable_mp(this, &ActionMapEditor::set_show_builtin_actions));
	add_hbox->add_child(show_builtin_actions_checkbutton);

	main_vbox->add_child(add_hbox);

	// Action Editor Tree
	action_tree = memnew(Tree);
	action_tree->set_v_size_flags(Control::SIZE_EXPAND_FILL);
	action_tree->set_columns(3);
	action_tree->set_hide_root(true);
	action_tree->set_column_titles_visible(true);
	action_tree->set_column_title(0, TTR("Action"));
	action_tree->set_column_clip_content(0, true);
	action_tree->set_column_title(1, TTR("Deadzone"));
	action_tree->set_column_expand(1, false);
	action_tree->set_column_custom_minimum_width(1, 80 * EDSCALE);
	action_tree->set_column_expand(2, false);
	action_tree->set_column_custom_minimum_width(2, 50 * EDSCALE);
	action_tree->connect("item_edited", callable_mp(this, &ActionMapEditor::_action_edited));
	action_tree->connect("item_activated", callable_mp(this, &ActionMapEditor::_tree_item_activated));
	action_tree->connect("button_clicked", callable_mp(this, &ActionMapEditor::_tree_button_pressed));
	main_vbox->add_child(action_tree);

	action_tree->set_drag_forwarding_compat(this);

	// Adding event dialog
	event_config_dialog = memnew(InputEventConfigurationDialog);
	event_config_dialog->connect("confirmed", callable_mp(this, &ActionMapEditor::_event_config_confirmed));
	add_child(event_config_dialog);

	message = memnew(AcceptDialog);
	add_child(message);
}
