/*************************************************************************/
/*  editor_profiler.h                                                    */
/*************************************************************************/
/*                       This file is part of:                           */
/*                           GODOT ENGINE                                */
/*                      https://godotengine.org                          */
/*************************************************************************/
/* Copyright (c) 2007-2022 Juan Linietsky, Ariel Manzur.                 */
/* Copyright (c) 2014-2022 Godot Engine contributors (cf. AUTHORS.md).   */
/*                                                                       */
/* Permission is hereby granted, free of charge, to any person obtaining */
/* a copy of this software and associated documentation files (the       */
/* "Software"), to deal in the Software without restriction, including   */
/* without limitation the rights to use, copy, modify, merge, publish,   */
/* distribute, sublicense, and/or sell copies of the Software, and to    */
/* permit persons to whom the Software is furnished to do so, subject to */
/* the following conditions:                                             */
/*                                                                       */
/* The above copyright notice and this permission notice shall be        */
/* included in all copies or substantial portions of the Software.       */
/*                                                                       */
/* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       */
/* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    */
/* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*/
/* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY  */
/* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,  */
/* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE     */
/* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                */
/*************************************************************************/

#ifndef EDITORPROFILER_H
#define EDITORPROFILER_H

#include "scene/gui/box_container.h"
#include "scene/gui/button.h"
#include "scene/gui/label.h"
#include "scene/gui/option_button.h"
#include "scene/gui/spin_box.h"
#include "scene/gui/split_container.h"
#include "scene/gui/texture_rect.h"
#include "scene/gui/tree.h"

class EditorProfiler : public VBoxContainer {
	GDCLASS(EditorProfiler, VBoxContainer);

public:
	struct Metric {
		bool valid = false;

		int frame_number = 0;
		float frame_time = 0;
		float idle_time = 0;
		float physics_time = 0;
		float physics_frame_time = 0;

		struct Category {
			StringName signature;
			String name;
			float total_time = 0; //total for category

			struct Item {
				StringName signature;
				String name;
				String script;
				int line = 0;
				float self = 0;
				float total = 0;
				int calls = 0;
			};

			Vector<Item> items;
		};

		Vector<Category> categories;

		Map<StringName, Category *> category_ptrs;
		Map<StringName, Category::Item *> item_ptrs;
	};

	enum DisplayMode {
		DISPLAY_FRAME_TIME,
		DISPLAY_AVERAGE_TIME,
		DISPLAY_FRAME_PERCENT,
		DISPLAY_PHYSICS_FRAME_PERCENT,
	};

	enum DisplayTime {
		DISPLAY_TOTAL_TIME,
		DISPLAY_SELF_TIME,
	};

private:
	Button *activate;
	Button *clear_button;
	TextureRect *graph;
	Ref<ImageTexture> graph_texture;
	Vector<uint8_t> graph_image;
	Tree *variables;
	HSplitContainer *h_split;

	Set<StringName> plot_sigs;

	OptionButton *display_mode;
	OptionButton *display_time;

	SpinBox *cursor_metric_edit;

	Vector<Metric> frame_metrics;
	int total_metrics;
	int last_metric;

	int max_functions;

	bool updating_frame;

	int hover_metric;

	float graph_height;

	bool seeking;

	Timer *frame_delay;
	Timer *plot_delay;

	void _update_frame();

	void _activate_pressed();
	void _clear_pressed();

	String _get_time_as_text(const Metric &m, float p_time, int p_calls);

	void _make_metric_ptrs(Metric &m);
	void _item_edited();

	void _update_plot();

	void _graph_tex_mouse_exit();

	void _graph_tex_draw();
	void _graph_tex_input(const Ref<InputEvent> &p_ev);

	Color _get_color_from_signature(const StringName &p_signature) const;

	void _cursor_metric_changed(double);

	void _combo_changed(int);

	Metric _get_frame_metric(int index);

protected:
	void _notification(int p_what);
	static void _bind_methods();

public:
	void add_frame_metric(const Metric &p_metric, bool p_final = false);
	void set_enabled(bool p_enable);
	bool is_profiling();
	bool is_seeking() { return seeking; }
	void disable_seeking();

	void clear();

	Vector<Vector<String>> get_data_as_csv() const;

	EditorProfiler();
};

#endif // EDITORPROFILER_H
