/*************************************************************************/
/*  core_constants.cpp                                                   */
/*************************************************************************/
/*                       This file is part of:                           */
/*                           GODOT ENGINE                                */
/*                      https://godotengine.org                          */
/*************************************************************************/
/* Copyright (c) 2007-2022 Juan Linietsky, Ariel Manzur.                 */
/* Copyright (c) 2014-2022 Godot Engine contributors (cf. AUTHORS.md).   */
/*                                                                       */
/* Permission is hereby granted, free of charge, to any person obtaining */
/* a copy of this software and associated documentation files (the       */
/* "Software"), to deal in the Software without restriction, including   */
/* without limitation the rights to use, copy, modify, merge, publish,   */
/* distribute, sublicense, and/or sell copies of the Software, and to    */
/* permit persons to whom the Software is furnished to do so, subject to */
/* the following conditions:                                             */
/*                                                                       */
/* The above copyright notice and this permission notice shall be        */
/* included in all copies or substantial portions of the Software.       */
/*                                                                       */
/* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       */
/* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    */
/* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*/
/* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY  */
/* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,  */
/* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE     */
/* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                */
/*************************************************************************/

#include "core_constants.h"

#include "core/input/input_event.h"
#include "core/multiplayer/multiplayer.h"
#include "core/object/class_db.h"
#include "core/os/keyboard.h"
#include "core/variant/variant.h"

struct _CoreConstant {
#ifdef DEBUG_METHODS_ENABLED
	StringName enum_name;
	bool ignore_value_in_docs = false;
#endif
	const char *name;
	int value = 0;

	_CoreConstant() {}

#ifdef DEBUG_METHODS_ENABLED
	_CoreConstant(const StringName &p_enum_name, const char *p_name, int p_value, bool p_ignore_value_in_docs = false) :
			enum_name(p_enum_name),
			ignore_value_in_docs(p_ignore_value_in_docs),
			name(p_name),
			value(p_value) {
	}
#else
	_CoreConstant(const char *p_name, int p_value) :
			name(p_name),
			value(p_value) {
	}
#endif
};

static Vector<_CoreConstant> _global_constants;

#ifdef DEBUG_METHODS_ENABLED

#define BIND_CORE_CONSTANT(m_constant) \
	_global_constants.push_back(_CoreConstant(StringName(), #m_constant, m_constant));

#define BIND_CORE_ENUM_CONSTANT(m_constant) \
	_global_constants.push_back(_CoreConstant(__constant_get_enum_name(m_constant, #m_constant), #m_constant, m_constant));

// This just binds enum classes as if they were regular enum constants.
#define BIND_CORE_ENUM_CLASS_CONSTANT(m_enum, m_prefix, m_member) \
	_global_constants.push_back(_CoreConstant(__constant_get_enum_name(m_enum::m_member, #m_prefix "_" #m_member), #m_prefix "_" #m_member, (int)m_enum::m_member));

#define BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(m_enum, m_name, m_member) \
	_global_constants.push_back(_CoreConstant(__constant_get_enum_name(m_enum::m_member, #m_name), #m_name, (int)m_enum::m_member));

#define BIND_CORE_ENUM_CLASS_CONSTANT_NO_VAL(m_enum, m_prefix, m_member) \
	_global_constants.push_back(_CoreConstant(__constant_get_enum_name(m_enum::m_member, #m_prefix "_" #m_member), #m_prefix "_" #m_member, (int)m_enum::m_member, true));

#define BIND_CORE_ENUM_CONSTANT_CUSTOM(m_custom_name, m_constant) \
	_global_constants.push_back(_CoreConstant(__constant_get_enum_name(m_constant, #m_constant), m_custom_name, m_constant));

#define BIND_CORE_CONSTANT_NO_VAL(m_constant) \
	_global_constants.push_back(_CoreConstant(StringName(), #m_constant, m_constant, true));

#define BIND_CORE_ENUM_CONSTANT_NO_VAL(m_constant) \
	_global_constants.push_back(_CoreConstant(__constant_get_enum_name(m_constant, #m_constant), #m_constant, m_constant, true));

#define BIND_CORE_ENUM_CONSTANT_CUSTOM_NO_VAL(m_custom_name, m_constant) \
	_global_constants.push_back(_CoreConstant(__constant_get_enum_name(m_constant, #m_constant), m_custom_name, m_constant, true));

#else

#define BIND_CORE_CONSTANT(m_constant) \
	_global_constants.push_back(_CoreConstant(#m_constant, m_constant));

#define BIND_CORE_ENUM_CONSTANT(m_constant) \
	_global_constants.push_back(_CoreConstant(#m_constant, m_constant));

// This just binds enum classes as if they were regular enum constants.
#define BIND_CORE_ENUM_CLASS_CONSTANT(m_enum, m_prefix, m_member) \
	_global_constants.push_back(_CoreConstant(#m_prefix "_" #m_member, (int)m_enum::m_member));

#define BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(m_enum, m_name, m_member) \
	_global_constants.push_back(_CoreConstant(#m_name, (int)m_enum::m_member));

#define BIND_CORE_ENUM_CLASS_CONSTANT_NO_VAL(m_enum, m_prefix, m_member) \
	_global_constants.push_back(_CoreConstant(#m_prefix "_" #m_member, (int)m_enum::m_member));

#define BIND_CORE_ENUM_CONSTANT_CUSTOM(m_custom_name, m_constant) \
	_global_constants.push_back(_CoreConstant(m_custom_name, m_constant));

#define BIND_CORE_CONSTANT_NO_VAL(m_constant) \
	_global_constants.push_back(_CoreConstant(#m_constant, m_constant));

#define BIND_CORE_ENUM_CONSTANT_NO_VAL(m_constant) \
	_global_constants.push_back(_CoreConstant(#m_constant, m_constant));

#define BIND_CORE_ENUM_CONSTANT_CUSTOM_NO_VAL(m_custom_name, m_constant) \
	_global_constants.push_back(_CoreConstant(m_custom_name, m_constant));

#endif

void register_global_constants() {
	BIND_CORE_ENUM_CONSTANT(SIDE_LEFT);
	BIND_CORE_ENUM_CONSTANT(SIDE_TOP);
	BIND_CORE_ENUM_CONSTANT(SIDE_RIGHT);
	BIND_CORE_ENUM_CONSTANT(SIDE_BOTTOM);

	BIND_CORE_ENUM_CONSTANT(CORNER_TOP_LEFT);
	BIND_CORE_ENUM_CONSTANT(CORNER_TOP_RIGHT);
	BIND_CORE_ENUM_CONSTANT(CORNER_BOTTOM_RIGHT);
	BIND_CORE_ENUM_CONSTANT(CORNER_BOTTOM_LEFT);

	BIND_CORE_ENUM_CONSTANT(VERTICAL);
	BIND_CORE_ENUM_CONSTANT(HORIZONTAL);

	BIND_CORE_ENUM_CONSTANT(CLOCKWISE);
	BIND_CORE_ENUM_CONSTANT(COUNTERCLOCKWISE);

	BIND_CORE_ENUM_CONSTANT(HORIZONTAL_ALIGNMENT_LEFT);
	BIND_CORE_ENUM_CONSTANT(HORIZONTAL_ALIGNMENT_CENTER);
	BIND_CORE_ENUM_CONSTANT(HORIZONTAL_ALIGNMENT_RIGHT);
	BIND_CORE_ENUM_CONSTANT(HORIZONTAL_ALIGNMENT_FILL);

	BIND_CORE_ENUM_CONSTANT(VERTICAL_ALIGNMENT_TOP);
	BIND_CORE_ENUM_CONSTANT(VERTICAL_ALIGNMENT_CENTER);
	BIND_CORE_ENUM_CONSTANT(VERTICAL_ALIGNMENT_BOTTOM);
	BIND_CORE_ENUM_CONSTANT(VERTICAL_ALIGNMENT_FILL);

	BIND_CORE_ENUM_CONSTANT(INLINE_ALIGNMENT_TOP_TO);
	BIND_CORE_ENUM_CONSTANT(INLINE_ALIGNMENT_CENTER_TO);
	BIND_CORE_ENUM_CONSTANT(INLINE_ALIGNMENT_BOTTOM_TO);

	BIND_CORE_ENUM_CONSTANT(INLINE_ALIGNMENT_TO_TOP);
	BIND_CORE_ENUM_CONSTANT(INLINE_ALIGNMENT_TO_CENTER);
	BIND_CORE_ENUM_CONSTANT(INLINE_ALIGNMENT_TO_BASELINE);
	BIND_CORE_ENUM_CONSTANT(INLINE_ALIGNMENT_TO_BOTTOM);

	BIND_CORE_ENUM_CONSTANT(INLINE_ALIGNMENT_TOP);
	BIND_CORE_ENUM_CONSTANT(INLINE_ALIGNMENT_CENTER);
	BIND_CORE_ENUM_CONSTANT(INLINE_ALIGNMENT_BOTTOM);

	BIND_CORE_ENUM_CONSTANT(INLINE_ALIGNMENT_IMAGE_MASK);
	BIND_CORE_ENUM_CONSTANT(INLINE_ALIGNMENT_TEXT_MASK);

	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, SPECIAL);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ESCAPE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, TAB);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BACKTAB);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BACKSPACE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ENTER);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_ENTER);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, INSERT);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(Key, KEY_DELETE, KEY_DELETE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, PAUSE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, PRINT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, SYSREQ);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, CLEAR);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, HOME);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, END);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LEFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, UP);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, RIGHT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, DOWN);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, PAGEUP);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, PAGEDOWN);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, SHIFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, CTRL);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, META);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ALT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, CAPSLOCK);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, NUMLOCK);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, SCROLLLOCK);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F1);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F2);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F3);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F4);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F5);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F6);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F7);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F8);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F9);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F10);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F11);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F12);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F13);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F14);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F15);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F16);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_MULTIPLY);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_DIVIDE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_SUBTRACT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_PERIOD);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_ADD);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_0);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_1);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_2);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_3);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_4);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_5);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_6);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_7);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_8);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, KP_9);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, SUPER_L);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, SUPER_R);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, MENU);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, HYPER_L);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, HYPER_R);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, HELP);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, DIRECTION_L);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, DIRECTION_R);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BACK);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, FORWARD);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, STOP);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, REFRESH);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, VOLUMEDOWN);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, VOLUMEMUTE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, VOLUMEUP);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BASSBOOST);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BASSUP);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BASSDOWN);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, TREBLEUP);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, TREBLEDOWN);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, MEDIAPLAY);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, MEDIASTOP);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, MEDIAPREVIOUS);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, MEDIANEXT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, MEDIARECORD);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, HOMEPAGE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, FAVORITES);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, SEARCH);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, STANDBY);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, OPENURL);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCHMAIL);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCHMEDIA);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCH0);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCH1);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCH2);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCH3);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCH4);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCH5);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCH6);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCH7);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCH8);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCH9);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCHA);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCHB);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCHC);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCHD);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCHE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LAUNCHF);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, UNKNOWN);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, SPACE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, EXCLAM);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, QUOTEDBL);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, NUMBERSIGN);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, DOLLAR);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, PERCENT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, AMPERSAND);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, APOSTROPHE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, PARENLEFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, PARENRIGHT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ASTERISK);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, PLUS);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, COMMA);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, MINUS);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, PERIOD);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, SLASH);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(Key, KEY_0, KEY_0);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(Key, KEY_1, KEY_1);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(Key, KEY_2, KEY_2);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(Key, KEY_3, KEY_3);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(Key, KEY_4, KEY_4);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(Key, KEY_5, KEY_5);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(Key, KEY_6, KEY_6);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(Key, KEY_7, KEY_7);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(Key, KEY_8, KEY_8);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(Key, KEY_9, KEY_9);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, COLON);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, SEMICOLON);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, LESS);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, EQUAL);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, GREATER);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, QUESTION);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, AT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, A);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, B);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, C);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, D);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, E);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, F);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, G);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, H);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, I);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, J);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, K);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, L);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, M);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, N);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, O);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, P);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, Q);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, R);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, S);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, T);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, U);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, V);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, W);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, X);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, Y);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, Z);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BRACKETLEFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BACKSLASH);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BRACKETRIGHT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ASCIICIRCUM);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, UNDERSCORE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, QUOTELEFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BRACELEFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BAR);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BRACERIGHT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ASCIITILDE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, NOBREAKSPACE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, EXCLAMDOWN);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, CENT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, STERLING);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, CURRENCY);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, YEN);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, BROKENBAR);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, SECTION);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, DIAERESIS);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, COPYRIGHT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ORDFEMININE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, GUILLEMOTLEFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, NOTSIGN);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, HYPHEN);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(Key, KEY_REGISTERED, KEY_REGISTERED);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, MACRON);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, DEGREE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, PLUSMINUS);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, TWOSUPERIOR);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, THREESUPERIOR);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ACUTE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, MU);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, PARAGRAPH);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, PERIODCENTERED);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, CEDILLA);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ONESUPERIOR);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, MASCULINE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, GUILLEMOTRIGHT);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ONEQUARTER);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ONEHALF);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, THREEQUARTERS);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, QUESTIONDOWN);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, AGRAVE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, AACUTE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ACIRCUMFLEX);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ATILDE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ADIAERESIS);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ARING);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, AE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, CCEDILLA);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, EGRAVE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, EACUTE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ECIRCUMFLEX);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, EDIAERESIS);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, IGRAVE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, IACUTE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ICIRCUMFLEX);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, IDIAERESIS);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ETH);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, NTILDE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, OGRAVE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, OACUTE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, OCIRCUMFLEX);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, OTILDE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, ODIAERESIS);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, MULTIPLY);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, OOBLIQUE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, UGRAVE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, UACUTE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, UCIRCUMFLEX);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, UDIAERESIS);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, YACUTE);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, THORN);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, SSHARP);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, DIVISION);
	BIND_CORE_ENUM_CLASS_CONSTANT(Key, KEY, YDIAERESIS);

	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(KeyModifierMask, KEY_CODE_MASK, CODE_MASK);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(KeyModifierMask, KEY_MODIFIER_MASK, MODIFIER_MASK);
	BIND_CORE_ENUM_CLASS_CONSTANT(KeyModifierMask, KEY_MASK, SHIFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(KeyModifierMask, KEY_MASK, ALT);
	BIND_CORE_ENUM_CLASS_CONSTANT(KeyModifierMask, KEY_MASK, META);
	BIND_CORE_ENUM_CLASS_CONSTANT(KeyModifierMask, KEY_MASK, CTRL);
	BIND_CORE_ENUM_CLASS_CONSTANT_NO_VAL(KeyModifierMask, KEY_MASK, CMD);
	BIND_CORE_ENUM_CLASS_CONSTANT(KeyModifierMask, KEY_MASK, KPAD);
	BIND_CORE_ENUM_CLASS_CONSTANT(KeyModifierMask, KEY_MASK, GROUP_SWITCH);

	BIND_CORE_ENUM_CLASS_CONSTANT(MouseButton, MOUSE_BUTTON, LEFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(MouseButton, MOUSE_BUTTON, RIGHT);
	BIND_CORE_ENUM_CLASS_CONSTANT(MouseButton, MOUSE_BUTTON, MIDDLE);
	BIND_CORE_ENUM_CLASS_CONSTANT(MouseButton, MOUSE_BUTTON, WHEEL_UP);
	BIND_CORE_ENUM_CLASS_CONSTANT(MouseButton, MOUSE_BUTTON, WHEEL_DOWN);
	BIND_CORE_ENUM_CLASS_CONSTANT(MouseButton, MOUSE_BUTTON, WHEEL_LEFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(MouseButton, MOUSE_BUTTON, WHEEL_RIGHT);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(MouseButton, MOUSE_BUTTON_XBUTTON1, MB_XBUTTON1);
	BIND_CORE_ENUM_CLASS_CONSTANT_CUSTOM(MouseButton, MOUSE_BUTTON_XBUTTON2, MB_XBUTTON2);
	BIND_CORE_ENUM_CLASS_CONSTANT(MouseButton, MOUSE_BUTTON, MASK_LEFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(MouseButton, MOUSE_BUTTON, MASK_RIGHT);
	BIND_CORE_ENUM_CLASS_CONSTANT(MouseButton, MOUSE_BUTTON, MASK_MIDDLE);
	BIND_CORE_ENUM_CLASS_CONSTANT(MouseButton, MOUSE_BUTTON, MASK_XBUTTON1);
	BIND_CORE_ENUM_CLASS_CONSTANT(MouseButton, MOUSE_BUTTON, MASK_XBUTTON2);

	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, INVALID);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, A);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, B);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, X);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, Y);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, BACK);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, GUIDE);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, START);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, LEFT_STICK);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, RIGHT_STICK);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, LEFT_SHOULDER);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, RIGHT_SHOULDER);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, DPAD_UP);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, DPAD_DOWN);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, DPAD_LEFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, DPAD_RIGHT);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, MISC1);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, PADDLE1);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, PADDLE2);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, PADDLE3);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, PADDLE4);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, TOUCHPAD);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, SDL_MAX);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyButton, JOY_BUTTON, MAX);

	BIND_CORE_ENUM_CLASS_CONSTANT(JoyAxis, JOY_AXIS, INVALID);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyAxis, JOY_AXIS, LEFT_X);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyAxis, JOY_AXIS, LEFT_Y);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyAxis, JOY_AXIS, RIGHT_X);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyAxis, JOY_AXIS, RIGHT_Y);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyAxis, JOY_AXIS, TRIGGER_LEFT);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyAxis, JOY_AXIS, TRIGGER_RIGHT);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyAxis, JOY_AXIS, SDL_MAX);
	BIND_CORE_ENUM_CLASS_CONSTANT(JoyAxis, JOY_AXIS, MAX);

	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, NOTE_OFF);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, NOTE_ON);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, AFTERTOUCH);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, CONTROL_CHANGE);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, PROGRAM_CHANGE);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, CHANNEL_PRESSURE);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, PITCH_BEND);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, SYSTEM_EXCLUSIVE);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, QUARTER_FRAME);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, SONG_POSITION_POINTER);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, SONG_SELECT);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, TUNE_REQUEST);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, TIMING_CLOCK);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, START);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, CONTINUE);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, STOP);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, ACTIVE_SENSING);
	BIND_CORE_ENUM_CLASS_CONSTANT(MIDIMessage, MIDI_MESSAGE, SYSTEM_RESET);

	// error list

	BIND_CORE_ENUM_CONSTANT(OK); // (0)
	BIND_CORE_ENUM_CONSTANT(FAILED);
	BIND_CORE_ENUM_CONSTANT(ERR_UNAVAILABLE);
	BIND_CORE_ENUM_CONSTANT(ERR_UNCONFIGURED);
	BIND_CORE_ENUM_CONSTANT(ERR_UNAUTHORIZED);
	BIND_CORE_ENUM_CONSTANT(ERR_PARAMETER_RANGE_ERROR); // (5)
	BIND_CORE_ENUM_CONSTANT(ERR_OUT_OF_MEMORY);
	BIND_CORE_ENUM_CONSTANT(ERR_FILE_NOT_FOUND);
	BIND_CORE_ENUM_CONSTANT(ERR_FILE_BAD_DRIVE);
	BIND_CORE_ENUM_CONSTANT(ERR_FILE_BAD_PATH);
	BIND_CORE_ENUM_CONSTANT(ERR_FILE_NO_PERMISSION); // (10)
	BIND_CORE_ENUM_CONSTANT(ERR_FILE_ALREADY_IN_USE);
	BIND_CORE_ENUM_CONSTANT(ERR_FILE_CANT_OPEN);
	BIND_CORE_ENUM_CONSTANT(ERR_FILE_CANT_WRITE);
	BIND_CORE_ENUM_CONSTANT(ERR_FILE_CANT_READ);
	BIND_CORE_ENUM_CONSTANT(ERR_FILE_UNRECOGNIZED); // (15)
	BIND_CORE_ENUM_CONSTANT(ERR_FILE_CORRUPT);
	BIND_CORE_ENUM_CONSTANT(ERR_FILE_MISSING_DEPENDENCIES);
	BIND_CORE_ENUM_CONSTANT(ERR_FILE_EOF);
	BIND_CORE_ENUM_CONSTANT(ERR_CANT_OPEN);
	BIND_CORE_ENUM_CONSTANT(ERR_CANT_CREATE); // (20)
	BIND_CORE_ENUM_CONSTANT(ERR_QUERY_FAILED);
	BIND_CORE_ENUM_CONSTANT(ERR_ALREADY_IN_USE);
	BIND_CORE_ENUM_CONSTANT(ERR_LOCKED);
	BIND_CORE_ENUM_CONSTANT(ERR_TIMEOUT);
	BIND_CORE_ENUM_CONSTANT(ERR_CANT_CONNECT); // (25)
	BIND_CORE_ENUM_CONSTANT(ERR_CANT_RESOLVE);
	BIND_CORE_ENUM_CONSTANT(ERR_CONNECTION_ERROR);
	BIND_CORE_ENUM_CONSTANT(ERR_CANT_ACQUIRE_RESOURCE);
	BIND_CORE_ENUM_CONSTANT(ERR_CANT_FORK);
	BIND_CORE_ENUM_CONSTANT(ERR_INVALID_DATA); // (30)
	BIND_CORE_ENUM_CONSTANT(ERR_INVALID_PARAMETER);
	BIND_CORE_ENUM_CONSTANT(ERR_ALREADY_EXISTS);
	BIND_CORE_ENUM_CONSTANT(ERR_DOES_NOT_EXIST);
	BIND_CORE_ENUM_CONSTANT(ERR_DATABASE_CANT_READ);
	BIND_CORE_ENUM_CONSTANT(ERR_DATABASE_CANT_WRITE); // (35)
	BIND_CORE_ENUM_CONSTANT(ERR_COMPILATION_FAILED);
	BIND_CORE_ENUM_CONSTANT(ERR_METHOD_NOT_FOUND);
	BIND_CORE_ENUM_CONSTANT(ERR_LINK_FAILED);
	BIND_CORE_ENUM_CONSTANT(ERR_SCRIPT_FAILED);
	BIND_CORE_ENUM_CONSTANT(ERR_CYCLIC_LINK); // (40)
	BIND_CORE_ENUM_CONSTANT(ERR_INVALID_DECLARATION);
	BIND_CORE_ENUM_CONSTANT(ERR_DUPLICATE_SYMBOL);
	BIND_CORE_ENUM_CONSTANT(ERR_PARSE_ERROR);
	BIND_CORE_ENUM_CONSTANT(ERR_BUSY);
	BIND_CORE_ENUM_CONSTANT(ERR_SKIP); // (45)
	BIND_CORE_ENUM_CONSTANT(ERR_HELP);
	BIND_CORE_ENUM_CONSTANT(ERR_BUG);
	BIND_CORE_ENUM_CONSTANT(ERR_PRINTER_ON_FIRE);

	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_NONE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_RANGE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_ENUM);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_ENUM_SUGGESTION);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_EXP_EASING);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_LENGTH);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_KEY_ACCEL);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_FLAGS);

	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_LAYERS_2D_RENDER);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_LAYERS_2D_PHYSICS);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_LAYERS_2D_NAVIGATION);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_LAYERS_3D_RENDER);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_LAYERS_3D_PHYSICS);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_LAYERS_3D_NAVIGATION);

	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_FILE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_DIR);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_GLOBAL_FILE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_GLOBAL_DIR);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_RESOURCE_TYPE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_MULTILINE_TEXT);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_PLACEHOLDER_TEXT);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_COLOR_NO_ALPHA);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_IMAGE_COMPRESS_LOSSY);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_IMAGE_COMPRESS_LOSSLESS);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_OBJECT_ID);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_TYPE_STRING);

	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_NODE_PATH_TO_EDITED_NODE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_METHOD_OF_VARIANT_TYPE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_METHOD_OF_BASE_TYPE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_METHOD_OF_INSTANCE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_METHOD_OF_SCRIPT);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_PROPERTY_OF_VARIANT_TYPE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_PROPERTY_OF_BASE_TYPE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_PROPERTY_OF_INSTANCE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_PROPERTY_OF_SCRIPT);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_OBJECT_TOO_BIG);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_NODE_PATH_VALID_TYPES);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_SAVE_FILE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_INT_IS_OBJECTID);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_INT_IS_POINTER);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_ARRAY_TYPE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_LOCALE_ID);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_HINT_MAX);

	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_NONE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_STORAGE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_EDITOR);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_NETWORK);

	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_EDITOR_HELPER);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_CHECKABLE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_CHECKED);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_INTERNATIONALIZED);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_GROUP);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_CATEGORY);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_SUBGROUP);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_NO_INSTANCE_STATE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_RESTART_IF_CHANGED);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_SCRIPT_VARIABLE);

	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_STORE_IF_NULL);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_ANIMATE_AS_TRIGGER);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_UPDATE_ALL_IF_MODIFIED);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_SCRIPT_DEFAULT_VALUE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_CLASS_IS_ENUM);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_NIL_IS_VARIANT);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_INTERNAL);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_DO_NOT_SHARE_ON_DUPLICATE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_HIGH_END_GFX);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_NODE_PATH_FROM_SCENE_ROOT);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_RESOURCE_NOT_PERSISTENT);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_KEYING_INCREMENTS);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_DEFERRED_SET_RESOURCE);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_EDITOR_INSTANTIATE_OBJECT);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_EDITOR_BASIC_SETTING);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_ARRAY);

	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_DEFAULT);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_DEFAULT_INTL);
	BIND_CORE_ENUM_CONSTANT(PROPERTY_USAGE_NO_EDITOR);

	BIND_CORE_ENUM_CONSTANT(METHOD_FLAG_NORMAL);
	BIND_CORE_ENUM_CONSTANT(METHOD_FLAG_EDITOR);
	BIND_CORE_ENUM_CONSTANT(METHOD_FLAG_NOSCRIPT);
	BIND_CORE_ENUM_CONSTANT(METHOD_FLAG_CONST);
	BIND_CORE_ENUM_CONSTANT(METHOD_FLAG_REVERSE);
	BIND_CORE_ENUM_CONSTANT(METHOD_FLAG_VIRTUAL);
	BIND_CORE_ENUM_CONSTANT(METHOD_FLAG_FROM_SCRIPT);
	BIND_CORE_ENUM_CONSTANT(METHOD_FLAG_STATIC);
	BIND_CORE_ENUM_CONSTANT(METHOD_FLAG_OBJECT_CORE);
	BIND_CORE_ENUM_CONSTANT(METHOD_FLAGS_DEFAULT);

	// rpc
	BIND_CORE_ENUM_CONSTANT_CUSTOM("RPC_MODE_DISABLED", Multiplayer::RPC_MODE_DISABLED);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("RPC_MODE_ANY_PEER", Multiplayer::RPC_MODE_ANY_PEER);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("RPC_MODE_AUTH", Multiplayer::RPC_MODE_AUTHORITY);

	BIND_CORE_ENUM_CONSTANT_CUSTOM("TRANSFER_MODE_UNRELIABLE", Multiplayer::TRANSFER_MODE_UNRELIABLE);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TRANSFER_MODE_UNRELIABLE_ORDERED", Multiplayer::TRANSFER_MODE_UNRELIABLE_ORDERED);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TRANSFER_MODE_RELIABLE", Multiplayer::TRANSFER_MODE_RELIABLE);

	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_NIL", Variant::NIL);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_BOOL", Variant::BOOL);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_INT", Variant::INT);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_FLOAT", Variant::FLOAT);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_STRING", Variant::STRING);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_VECTOR2", Variant::VECTOR2);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_VECTOR2I", Variant::VECTOR2I);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_RECT2", Variant::RECT2);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_RECT2I", Variant::RECT2I);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_VECTOR3", Variant::VECTOR3);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_VECTOR3I", Variant::VECTOR3I);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_TRANSFORM2D", Variant::TRANSFORM2D);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_PLANE", Variant::PLANE);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_QUATERNION", Variant::QUATERNION);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_AABB", Variant::AABB);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_BASIS", Variant::BASIS);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_TRANSFORM3D", Variant::TRANSFORM3D);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_COLOR", Variant::COLOR);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_STRING_NAME", Variant::STRING_NAME);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_NODE_PATH", Variant::NODE_PATH);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_RID", Variant::RID);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_OBJECT", Variant::OBJECT);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_CALLABLE", Variant::CALLABLE);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_SIGNAL", Variant::SIGNAL);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_DICTIONARY", Variant::DICTIONARY);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_ARRAY", Variant::ARRAY);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_PACKED_BYTE_ARRAY", Variant::PACKED_BYTE_ARRAY);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_PACKED_INT32_ARRAY", Variant::PACKED_INT32_ARRAY);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_PACKED_INT64_ARRAY", Variant::PACKED_INT64_ARRAY);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_PACKED_FLOAT32_ARRAY", Variant::PACKED_FLOAT32_ARRAY);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_PACKED_FLOAT64_ARRAY", Variant::PACKED_FLOAT64_ARRAY);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_PACKED_STRING_ARRAY", Variant::PACKED_STRING_ARRAY);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_PACKED_VECTOR2_ARRAY", Variant::PACKED_VECTOR2_ARRAY);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_PACKED_VECTOR3_ARRAY", Variant::PACKED_VECTOR3_ARRAY);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_PACKED_COLOR_ARRAY", Variant::PACKED_COLOR_ARRAY);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("TYPE_MAX", Variant::VARIANT_MAX);

	//comparison
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_EQUAL", Variant::OP_EQUAL);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_NOT_EQUAL", Variant::OP_NOT_EQUAL);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_LESS", Variant::OP_LESS);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_LESS_EQUAL", Variant::OP_LESS_EQUAL);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_GREATER", Variant::OP_GREATER);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_GREATER_EQUAL", Variant::OP_GREATER_EQUAL);
	//mathematic
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_ADD", Variant::OP_ADD);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_SUBTRACT", Variant::OP_SUBTRACT);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_MULTIPLY", Variant::OP_MULTIPLY);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_DIVIDE", Variant::OP_DIVIDE);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_NEGATE", Variant::OP_NEGATE);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_POSITIVE", Variant::OP_POSITIVE);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_MODULE", Variant::OP_MODULE);
	//bitwise
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_SHIFT_LEFT", Variant::OP_SHIFT_LEFT);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_SHIFT_RIGHT", Variant::OP_SHIFT_RIGHT);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_BIT_AND", Variant::OP_BIT_AND);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_BIT_OR", Variant::OP_BIT_OR);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_BIT_XOR", Variant::OP_BIT_XOR);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_BIT_NEGATE", Variant::OP_BIT_NEGATE);
	//logic
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_AND", Variant::OP_AND);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_OR", Variant::OP_OR);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_XOR", Variant::OP_XOR);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_NOT", Variant::OP_NOT);
	//containment
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_IN", Variant::OP_IN);
	BIND_CORE_ENUM_CONSTANT_CUSTOM("OP_MAX", Variant::OP_MAX);
}

void unregister_global_constants() {
	_global_constants.clear();
}

int CoreConstants::get_global_constant_count() {
	return _global_constants.size();
}

#ifdef DEBUG_METHODS_ENABLED
StringName CoreConstants::get_global_constant_enum(int p_idx) {
	return _global_constants[p_idx].enum_name;
}

bool CoreConstants::get_ignore_value_in_docs(int p_idx) {
	return _global_constants[p_idx].ignore_value_in_docs;
}
#else
StringName CoreConstants::get_global_constant_enum(int p_idx) {
	return StringName();
}

bool CoreConstants::get_ignore_value_in_docs(int p_idx) {
	return false;
}
#endif

const char *CoreConstants::get_global_constant_name(int p_idx) {
	return _global_constants[p_idx].name;
}

int CoreConstants::get_global_constant_value(int p_idx) {
	return _global_constants[p_idx].value;
}
