/*************************************************************************/
/*  project_manager.cpp                                                  */
/*************************************************************************/
/*                       This file is part of:                           */
/*                           GODOT ENGINE                                */
/*                      https://godotengine.org                          */
/*************************************************************************/
/* Copyright (c) 2007-2022 Juan Linietsky, Ariel Manzur.                 */
/* Copyright (c) 2014-2022 Godot Engine contributors (cf. AUTHORS.md).   */
/*                                                                       */
/* Permission is hereby granted, free of charge, to any person obtaining */
/* a copy of this software and associated documentation files (the       */
/* "Software"), to deal in the Software without restriction, including   */
/* without limitation the rights to use, copy, modify, merge, publish,   */
/* distribute, sublicense, and/or sell copies of the Software, and to    */
/* permit persons to whom the Software is furnished to do so, subject to */
/* the following conditions:                                             */
/*                                                                       */
/* The above copyright notice and this permission notice shall be        */
/* included in all copies or substantial portions of the Software.       */
/*                                                                       */
/* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       */
/* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    */
/* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*/
/* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY  */
/* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,  */
/* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE     */
/* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                */
/*************************************************************************/

#include "project_manager.h"

#include "core/config/project_settings.h"
#include "core/io/config_file.h"
#include "core/io/dir_access.h"
#include "core/io/file_access.h"
#include "core/io/resource_saver.h"
#include "core/io/stream_peer_tls.h"
#include "core/io/zip_io.h"
#include "core/os/keyboard.h"
#include "core/os/os.h"
#include "core/string/translation.h"
#include "core/version.h"
#include "editor/editor_file_dialog.h"
#include "editor/editor_paths.h"
#include "editor/editor_scale.h"
#include "editor/editor_settings.h"
#include "editor/editor_themes.h"
#include "editor/editor_vcs_interface.h"
#include "main/main.h"
#include "scene/gui/center_container.h"
#include "scene/gui/line_edit.h"
#include "scene/gui/margin_container.h"
#include "scene/gui/panel_container.h"
#include "scene/gui/separator.h"
#include "scene/gui/texture_rect.h"
#include "scene/main/window.h"
#include "servers/display_server.h"
#include "servers/navigation_server_3d.h"
#include "servers/physics_server_2d.h"

constexpr int GODOT4_CONFIG_VERSION = 5;

class ProjectDialog : public ConfirmationDialog {
	GDCLASS(ProjectDialog, ConfirmationDialog);

public:
	bool is_folder_empty = true;
	enum Mode {
		MODE_NEW,
		MODE_IMPORT,
		MODE_INSTALL,
		MODE_RENAME,
	};

private:
	enum MessageType {
		MESSAGE_ERROR,
		MESSAGE_WARNING,
		MESSAGE_SUCCESS,
	};

	enum InputType {
		PROJECT_PATH,
		INSTALL_PATH,
	};

	Mode mode;
	Button *browse;
	Button *install_browse;
	Button *create_dir;
	Container *name_container;
	Container *path_container;
	Container *install_path_container;
	Container *rasterizer_container;
	HBoxContainer *default_files_container;
	Ref<ButtonGroup> rasterizer_button_group;
	Label *msg;
	LineEdit *project_path;
	LineEdit *project_name;
	LineEdit *install_path;
	TextureRect *status_rect;
	TextureRect *install_status_rect;
	EditorFileDialog *fdialog;
	EditorFileDialog *fdialog_install;
	OptionButton *vcs_metadata_selection;
	String zip_path;
	String zip_title;
	AcceptDialog *dialog_error;
	String fav_dir;

	String created_folder_path;

	void set_message(const String &p_msg, MessageType p_type = MESSAGE_SUCCESS, InputType input_type = PROJECT_PATH) {
		msg->set_text(p_msg);
		Ref<Texture2D> current_path_icon = status_rect->get_texture();
		Ref<Texture2D> current_install_icon = install_status_rect->get_texture();
		Ref<Texture2D> new_icon;

		switch (p_type) {
			case MESSAGE_ERROR: {
				msg->add_theme_color_override("font_color", msg->get_theme_color(SNAME("error_color"), SNAME("Editor")));
				msg->set_modulate(Color(1, 1, 1, 1));
				new_icon = msg->get_theme_icon(SNAME("StatusError"), SNAME("EditorIcons"));

			} break;
			case MESSAGE_WARNING: {
				msg->add_theme_color_override("font_color", msg->get_theme_color(SNAME("warning_color"), SNAME("Editor")));
				msg->set_modulate(Color(1, 1, 1, 1));
				new_icon = msg->get_theme_icon(SNAME("StatusWarning"), SNAME("EditorIcons"));

			} break;
			case MESSAGE_SUCCESS: {
				msg->set_modulate(Color(1, 1, 1, 0));
				new_icon = msg->get_theme_icon(SNAME("StatusSuccess"), SNAME("EditorIcons"));

			} break;
		}

		if (current_path_icon != new_icon && input_type == PROJECT_PATH) {
			status_rect->set_texture(new_icon);
		} else if (current_install_icon != new_icon && input_type == INSTALL_PATH) {
			install_status_rect->set_texture(new_icon);
		}

		set_size(Size2(500, 0) * EDSCALE);
	}

	String _test_path() {
		Ref<DirAccess> d = DirAccess::create(DirAccess::ACCESS_FILESYSTEM);
		String valid_path, valid_install_path;
		if (d->change_dir(project_path->get_text()) == OK) {
			valid_path = project_path->get_text();
		} else if (d->change_dir(project_path->get_text().strip_edges()) == OK) {
			valid_path = project_path->get_text().strip_edges();
		} else if (project_path->get_text().ends_with(".zip")) {
			if (d->file_exists(project_path->get_text())) {
				valid_path = project_path->get_text();
			}
		} else if (project_path->get_text().strip_edges().ends_with(".zip")) {
			if (d->file_exists(project_path->get_text().strip_edges())) {
				valid_path = project_path->get_text().strip_edges();
			}
		}

		if (valid_path.is_empty()) {
			set_message(TTR("The path specified doesn't exist."), MESSAGE_ERROR);
			get_ok_button()->set_disabled(true);
			return "";
		}

		if (mode == MODE_IMPORT && valid_path.ends_with(".zip")) {
			if (d->change_dir(install_path->get_text()) == OK) {
				valid_install_path = install_path->get_text();
			} else if (d->change_dir(install_path->get_text().strip_edges()) == OK) {
				valid_install_path = install_path->get_text().strip_edges();
			}

			if (valid_install_path.is_empty()) {
				set_message(TTR("The path specified doesn't exist."), MESSAGE_ERROR, INSTALL_PATH);
				get_ok_button()->set_disabled(true);
				return "";
			}
		}

		if (mode == MODE_IMPORT || mode == MODE_RENAME) {
			if (!valid_path.is_empty() && !d->file_exists("project.godot")) {
				if (valid_path.ends_with(".zip")) {
					Ref<FileAccess> io_fa;
					zlib_filefunc_def io = zipio_create_io(&io_fa);

					unzFile pkg = unzOpen2(valid_path.utf8().get_data(), &io);
					if (!pkg) {
						set_message(TTR("Error opening package file (it's not in ZIP format)."), MESSAGE_ERROR);
						get_ok_button()->set_disabled(true);
						unzClose(pkg);
						return "";
					}

					int ret = unzGoToFirstFile(pkg);
					while (ret == UNZ_OK) {
						unz_file_info info;
						char fname[16384];
						ret = unzGetCurrentFileInfo(pkg, &info, fname, 16384, nullptr, 0, nullptr, 0);
						if (ret != UNZ_OK) {
							break;
						}

						if (String::utf8(fname).ends_with("project.godot")) {
							break;
						}

						ret = unzGoToNextFile(pkg);
					}

					if (ret == UNZ_END_OF_LIST_OF_FILE) {
						set_message(TTR("Invalid \".zip\" project file; it doesn't contain a \"project.godot\" file."), MESSAGE_ERROR);
						get_ok_button()->set_disabled(true);
						unzClose(pkg);
						return "";
					}

					unzClose(pkg);

					// check if the specified install folder is empty, even though this is not an error, it is good to check here
					d->list_dir_begin();
					is_folder_empty = true;
					String n = d->get_next();
					while (!n.is_empty()) {
						if (!n.begins_with(".")) {
							// Allow `.`, `..` (reserved current/parent folder names)
							// and hidden files/folders to be present.
							// For instance, this lets users initialize a Git repository
							// and still be able to create a project in the directory afterwards.
							is_folder_empty = false;
							break;
						}
						n = d->get_next();
					}
					d->list_dir_end();

					if (!is_folder_empty) {
						set_message(TTR("Please choose an empty folder."), MESSAGE_WARNING, INSTALL_PATH);
						get_ok_button()->set_disabled(true);
						return "";
					}

				} else {
					set_message(TTR("Please choose a \"project.godot\" or \".zip\" file."), MESSAGE_ERROR);
					install_path_container->hide();
					get_ok_button()->set_disabled(true);
					return "";
				}

			} else if (valid_path.ends_with("zip")) {
				set_message(TTR("This directory already contains a Godot project."), MESSAGE_ERROR, INSTALL_PATH);
				get_ok_button()->set_disabled(true);
				return "";
			}

		} else {
			// check if the specified folder is empty, even though this is not an error, it is good to check here
			d->list_dir_begin();
			is_folder_empty = true;
			String n = d->get_next();
			while (!n.is_empty()) {
				if (!n.begins_with(".")) {
					// Allow `.`, `..` (reserved current/parent folder names)
					// and hidden files/folders to be present.
					// For instance, this lets users initialize a Git repository
					// and still be able to create a project in the directory afterwards.
					is_folder_empty = false;
					break;
				}
				n = d->get_next();
			}
			d->list_dir_end();

			if (!is_folder_empty) {
				set_message(TTR("The selected path is not empty. Choosing an empty folder is highly recommended."), MESSAGE_WARNING);
				get_ok_button()->set_disabled(false);
				return valid_path;
			}
		}

		set_message("");
		set_message("", MESSAGE_SUCCESS, INSTALL_PATH);
		get_ok_button()->set_disabled(false);
		return valid_path;
	}

	void _path_text_changed(const String &p_path) {
		String sp = _test_path();
		if (!sp.is_empty()) {
			// If the project name is empty or default, infer the project name from the selected folder name
			if (project_name->get_text().strip_edges().is_empty() || project_name->get_text().strip_edges() == TTR("New Game Project")) {
				sp = sp.replace("\\", "/");
				int lidx = sp.rfind("/");

				if (lidx != -1) {
					sp = sp.substr(lidx + 1, sp.length()).capitalize();
				}
				if (sp.is_empty() && mode == MODE_IMPORT) {
					sp = TTR("Imported Project");
				}

				project_name->set_text(sp);
				_text_changed(sp);
			}
		}

		if (!created_folder_path.is_empty() && created_folder_path != p_path) {
			_remove_created_folder();
		}
	}

	void _file_selected(const String &p_path) {
		String p = p_path;
		if (mode == MODE_IMPORT) {
			if (p.ends_with("project.godot")) {
				p = p.get_base_dir();
				install_path_container->hide();
				get_ok_button()->set_disabled(false);
			} else if (p.ends_with(".zip")) {
				install_path->set_text(p.get_base_dir());
				install_path_container->show();
				get_ok_button()->set_disabled(false);
			} else {
				set_message(TTR("Please choose a \"project.godot\" or \".zip\" file."), MESSAGE_ERROR);
				get_ok_button()->set_disabled(true);
				return;
			}
		}

		String sp = p.simplify_path();
		project_path->set_text(sp);
		_path_text_changed(sp);
		if (p.ends_with(".zip")) {
			install_path->call_deferred(SNAME("grab_focus"));
		} else {
			get_ok_button()->call_deferred(SNAME("grab_focus"));
		}
	}

	void _path_selected(const String &p_path) {
		String sp = p_path.simplify_path();
		project_path->set_text(sp);
		_path_text_changed(sp);
		get_ok_button()->call_deferred(SNAME("grab_focus"));
	}

	void _install_path_selected(const String &p_path) {
		String sp = p_path.simplify_path();
		install_path->set_text(sp);
		_path_text_changed(sp);
		get_ok_button()->call_deferred(SNAME("grab_focus"));
	}

	void _browse_path() {
		fdialog->set_current_dir(project_path->get_text());

		if (mode == MODE_IMPORT) {
			fdialog->set_file_mode(EditorFileDialog::FILE_MODE_OPEN_FILE);
			fdialog->clear_filters();
			fdialog->add_filter("project.godot", vformat("%s %s", VERSION_NAME, TTR("Project")));
			fdialog->add_filter("*.zip", TTR("ZIP File"));
		} else {
			fdialog->set_file_mode(EditorFileDialog::FILE_MODE_OPEN_DIR);
		}
		fdialog->popup_file_dialog();
	}

	void _browse_install_path() {
		fdialog_install->set_current_dir(install_path->get_text());
		fdialog_install->set_file_mode(EditorFileDialog::FILE_MODE_OPEN_DIR);
		fdialog_install->popup_file_dialog();
	}

	void _create_folder() {
		const String project_name_no_edges = project_name->get_text().strip_edges();
		if (project_name_no_edges.is_empty() || !created_folder_path.is_empty() || project_name_no_edges.ends_with(".")) {
			set_message(TTR("Invalid project name."), MESSAGE_WARNING);
			return;
		}

		Ref<DirAccess> d = DirAccess::create(DirAccess::ACCESS_FILESYSTEM);
		if (d->change_dir(project_path->get_text()) == OK) {
			if (!d->dir_exists(project_name_no_edges)) {
				if (d->make_dir(project_name_no_edges) == OK) {
					d->change_dir(project_name_no_edges);
					String dir_str = d->get_current_dir();
					project_path->set_text(dir_str);
					_path_text_changed(dir_str);
					created_folder_path = d->get_current_dir();
					create_dir->set_disabled(true);
				} else {
					dialog_error->set_text(TTR("Couldn't create folder."));
					dialog_error->popup_centered();
				}
			} else {
				dialog_error->set_text(TTR("There is already a folder in this path with the specified name."));
				dialog_error->popup_centered();
			}
		}
	}

	void _text_changed(const String &p_text) {
		if (mode != MODE_NEW) {
			return;
		}

		_test_path();

		if (p_text.strip_edges().is_empty()) {
			set_message(TTR("It would be a good idea to name your project."), MESSAGE_ERROR);
		}
	}

	void _nonempty_confirmation_ok_pressed() {
		is_folder_empty = true;
		ok_pressed();
	}

	void ok_pressed() override {
		String dir = project_path->get_text();

		if (mode == MODE_RENAME) {
			String dir2 = _test_path();
			if (dir2.is_empty()) {
				set_message(TTR("Invalid project path (changed anything?)."), MESSAGE_ERROR);
				return;
			}

			ProjectSettings *current = memnew(ProjectSettings);

			int err = current->setup(dir2, "");
			if (err != OK) {
				set_message(vformat(TTR("Couldn't load project.godot in project path (error %d). It may be missing or corrupted."), err), MESSAGE_ERROR);
			} else {
				ProjectSettings::CustomMap edited_settings;
				edited_settings["application/config/name"] = project_name->get_text().strip_edges();

				if (current->save_custom(dir2.path_join("project.godot"), edited_settings, Vector<String>(), true) != OK) {
					set_message(TTR("Couldn't edit project.godot in project path."), MESSAGE_ERROR);
				}
			}

			hide();
			emit_signal(SNAME("projects_updated"));

		} else {
			if (mode == MODE_IMPORT) {
				if (project_path->get_text().ends_with(".zip")) {
					mode = MODE_INSTALL;
					ok_pressed();

					return;
				}

			} else {
				if (mode == MODE_NEW) {
					// Before we create a project, check that the target folder is empty.
					// If not, we need to ask the user if they're sure they want to do this.
					if (!is_folder_empty) {
						ConfirmationDialog *cd = memnew(ConfirmationDialog);
						cd->set_title(TTR("Warning: This folder is not empty"));
						cd->set_text(TTR("You are about to create a Godot project in a non-empty folder.\nThe entire contents of this folder will be imported as project resources!\n\nAre you sure you wish to continue?"));
						cd->get_ok_button()->connect("pressed", callable_mp(this, &ProjectDialog::_nonempty_confirmation_ok_pressed));
						get_parent()->add_child(cd);
						cd->popup_centered();
						cd->grab_focus();
						return;
					}
					PackedStringArray project_features = ProjectSettings::get_required_features();
					ProjectSettings::CustomMap initial_settings;
					// Be sure to change this code if/when renderers are changed.
					int renderer_type = rasterizer_button_group->get_pressed_button()->get_meta(SNAME("driver_name"));
					initial_settings["rendering/vulkan/rendering/back_end"] = renderer_type;
					if (renderer_type == 0) {
						project_features.push_back("Vulkan Clustered");
					} else if (renderer_type == 1) {
						project_features.push_back("Vulkan Mobile");
					} else {
						WARN_PRINT("Unknown renderer type. Please report this as a bug on GitHub.");
					}
					project_features.sort();
					initial_settings["application/config/features"] = project_features;
					initial_settings["application/config/name"] = project_name->get_text().strip_edges();
					initial_settings["application/config/icon"] = "res://icon.svg";

					if (ProjectSettings::get_singleton()->save_custom(dir.path_join("project.godot"), initial_settings, Vector<String>(), false) != OK) {
						set_message(TTR("Couldn't create project.godot in project path."), MESSAGE_ERROR);
					} else {
						// Store default project icon in SVG format.
						Error err;
						Ref<FileAccess> fa_icon = FileAccess::open(dir.path_join("icon.svg"), FileAccess::WRITE, &err);
						fa_icon->store_string(get_default_project_icon());

						if (err != OK) {
							set_message(TTR("Couldn't create icon.svg in project path."), MESSAGE_ERROR);
						}

						EditorVCSInterface::create_vcs_metadata_files(EditorVCSInterface::VCSMetadata(vcs_metadata_selection->get_selected()), dir);
					}
				} else if (mode == MODE_INSTALL) {
					if (project_path->get_text().ends_with(".zip")) {
						dir = install_path->get_text();
						zip_path = project_path->get_text();
					}

					Ref<FileAccess> io_fa;
					zlib_filefunc_def io = zipio_create_io(&io_fa);

					unzFile pkg = unzOpen2(zip_path.utf8().get_data(), &io);
					if (!pkg) {
						dialog_error->set_text(TTR("Error opening package file, not in ZIP format."));
						dialog_error->popup_centered();
						return;
					}

					// Find the zip_root
					String zip_root;
					int ret = unzGoToFirstFile(pkg);
					while (ret == UNZ_OK) {
						unz_file_info info;
						char fname[16384];
						unzGetCurrentFileInfo(pkg, &info, fname, 16384, nullptr, 0, nullptr, 0);

						String name = String::utf8(fname);
						if (name.ends_with("project.godot")) {
							zip_root = name.substr(0, name.rfind("project.godot"));
							break;
						}

						ret = unzGoToNextFile(pkg);
					}

					ret = unzGoToFirstFile(pkg);

					Vector<String> failed_files;

					int idx = 0;
					while (ret == UNZ_OK) {
						//get filename
						unz_file_info info;
						char fname[16384];
						ret = unzGetCurrentFileInfo(pkg, &info, fname, 16384, nullptr, 0, nullptr, 0);
						if (ret != UNZ_OK) {
							break;
						}

						String path = String::utf8(fname);

						if (path.is_empty() || path == zip_root || !zip_root.is_subsequence_of(path)) {
							//
						} else if (path.ends_with("/")) { // a dir
							path = path.substr(0, path.length() - 1);
							String rel_path = path.substr(zip_root.length());

							Ref<DirAccess> da = DirAccess::create(DirAccess::ACCESS_FILESYSTEM);
							da->make_dir(dir.path_join(rel_path));
						} else {
							Vector<uint8_t> data;
							data.resize(info.uncompressed_size);
							String rel_path = path.substr(zip_root.length());

							//read
							unzOpenCurrentFile(pkg);
							ret = unzReadCurrentFile(pkg, data.ptrw(), data.size());
							ERR_BREAK_MSG(ret < 0, vformat("An error occurred while attempting to read from file: %s. This file will not be used.", rel_path));
							unzCloseCurrentFile(pkg);

							Ref<FileAccess> f = FileAccess::open(dir.path_join(rel_path), FileAccess::WRITE);
							if (f.is_valid()) {
								f->store_buffer(data.ptr(), data.size());
							} else {
								failed_files.push_back(rel_path);
							}
						}

						idx++;
						ret = unzGoToNextFile(pkg);
					}

					unzClose(pkg);

					if (failed_files.size()) {
						String err_msg = TTR("The following files failed extraction from package:") + "\n\n";
						for (int i = 0; i < failed_files.size(); i++) {
							if (i > 15) {
								err_msg += "\nAnd " + itos(failed_files.size() - i) + " more files.";
								break;
							}
							err_msg += failed_files[i] + "\n";
						}

						dialog_error->set_text(err_msg);
						dialog_error->popup_centered();

					} else if (!project_path->get_text().ends_with(".zip")) {
						dialog_error->set_text(TTR("Package installed successfully!"));
						dialog_error->popup_centered();
					}
				}
			}

			dir = dir.replace("\\", "/");
			if (dir.ends_with("/")) {
				dir = dir.substr(0, dir.length() - 1);
			}

			hide();
			emit_signal(SNAME("project_created"), dir);
		}
	}

	void _remove_created_folder() {
		if (!created_folder_path.is_empty()) {
			Ref<DirAccess> d = DirAccess::create(DirAccess::ACCESS_FILESYSTEM);
			d->remove(created_folder_path);

			create_dir->set_disabled(false);
			created_folder_path = "";
		}
	}

	void cancel_pressed() override {
		_remove_created_folder();

		project_path->clear();
		_path_text_changed("");
		project_name->clear();
		_text_changed("");

		if (status_rect->get_texture() == msg->get_theme_icon(SNAME("StatusError"), SNAME("EditorIcons"))) {
			msg->show();
		}

		if (install_status_rect->get_texture() == msg->get_theme_icon(SNAME("StatusError"), SNAME("EditorIcons"))) {
			msg->show();
		}
	}

	void _notification(int p_what) {
		switch (p_what) {
			case NOTIFICATION_WM_CLOSE_REQUEST: {
				_remove_created_folder();
			} break;
		}
	}

protected:
	static void _bind_methods() {
		ADD_SIGNAL(MethodInfo("project_created"));
		ADD_SIGNAL(MethodInfo("projects_updated"));
	}

public:
	void set_zip_path(const String &p_path) {
		zip_path = p_path;
	}
	void set_zip_title(const String &p_title) {
		zip_title = p_title;
	}

	void set_mode(Mode p_mode) {
		mode = p_mode;
	}

	void set_project_path(const String &p_path) {
		project_path->set_text(p_path);
	}

	void show_dialog() {
		if (mode == MODE_RENAME) {
			project_path->set_editable(false);
			browse->hide();
			install_browse->hide();

			set_title(TTR("Rename Project"));
			set_ok_button_text(TTR("Rename"));
			name_container->show();
			status_rect->hide();
			msg->hide();
			install_path_container->hide();
			install_status_rect->hide();
			rasterizer_container->hide();
			default_files_container->hide();
			get_ok_button()->set_disabled(false);

			ProjectSettings *current = memnew(ProjectSettings);

			int err = current->setup(project_path->get_text(), "");
			if (err != OK) {
				set_message(vformat(TTR("Couldn't load project.godot in project path (error %d). It may be missing or corrupted."), err), MESSAGE_ERROR);
				status_rect->show();
				msg->show();
				get_ok_button()->set_disabled(true);
			} else if (current->has_setting("application/config/name")) {
				String proj = current->get("application/config/name");
				project_name->set_text(proj);
				_text_changed(proj);
			}

			project_name->call_deferred(SNAME("grab_focus"));

			create_dir->hide();

		} else {
			fav_dir = EditorSettings::get_singleton()->get("filesystem/directories/default_project_path");
			if (!fav_dir.is_empty()) {
				project_path->set_text(fav_dir);
				fdialog->set_current_dir(fav_dir);
			} else {
				Ref<DirAccess> d = DirAccess::create(DirAccess::ACCESS_FILESYSTEM);
				project_path->set_text(d->get_current_dir());
				fdialog->set_current_dir(d->get_current_dir());
			}
			String proj = TTR("New Game Project");
			project_name->set_text(proj);
			_text_changed(proj);

			project_path->set_editable(true);
			browse->set_disabled(false);
			browse->show();
			install_browse->set_disabled(false);
			install_browse->show();
			create_dir->show();
			status_rect->show();
			install_status_rect->show();
			msg->show();

			if (mode == MODE_IMPORT) {
				set_title(TTR("Import Existing Project"));
				set_ok_button_text(TTR("Import & Edit"));
				name_container->hide();
				install_path_container->hide();
				rasterizer_container->hide();
				default_files_container->hide();
				project_path->grab_focus();

			} else if (mode == MODE_NEW) {
				set_title(TTR("Create New Project"));
				set_ok_button_text(TTR("Create & Edit"));
				name_container->show();
				install_path_container->hide();
				rasterizer_container->show();
				default_files_container->show();
				project_name->call_deferred(SNAME("grab_focus"));
				project_name->call_deferred(SNAME("select_all"));

			} else if (mode == MODE_INSTALL) {
				set_title(TTR("Install Project:") + " " + zip_title);
				set_ok_button_text(TTR("Install & Edit"));
				project_name->set_text(zip_title);
				name_container->show();
				install_path_container->hide();
				rasterizer_container->hide();
				default_files_container->hide();
				project_path->grab_focus();
			}

			_test_path();
		}

		popup_centered(Size2(500, 0) * EDSCALE);
	}

	ProjectDialog() {
		VBoxContainer *vb = memnew(VBoxContainer);
		add_child(vb);

		name_container = memnew(VBoxContainer);
		vb->add_child(name_container);

		Label *l = memnew(Label);
		l->set_text(TTR("Project Name:"));
		name_container->add_child(l);

		HBoxContainer *pnhb = memnew(HBoxContainer);
		name_container->add_child(pnhb);

		project_name = memnew(LineEdit);
		project_name->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		pnhb->add_child(project_name);

		create_dir = memnew(Button);
		pnhb->add_child(create_dir);
		create_dir->set_text(TTR("Create Folder"));
		create_dir->connect("pressed", callable_mp(this, &ProjectDialog::_create_folder));

		path_container = memnew(VBoxContainer);
		vb->add_child(path_container);

		l = memnew(Label);
		l->set_text(TTR("Project Path:"));
		path_container->add_child(l);

		HBoxContainer *pphb = memnew(HBoxContainer);
		path_container->add_child(pphb);

		project_path = memnew(LineEdit);
		project_path->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		project_path->set_structured_text_bidi_override(TextServer::STRUCTURED_TEXT_FILE);
		pphb->add_child(project_path);

		install_path_container = memnew(VBoxContainer);
		vb->add_child(install_path_container);

		l = memnew(Label);
		l->set_text(TTR("Project Installation Path:"));
		install_path_container->add_child(l);

		HBoxContainer *iphb = memnew(HBoxContainer);
		install_path_container->add_child(iphb);

		install_path = memnew(LineEdit);
		install_path->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		install_path->set_structured_text_bidi_override(TextServer::STRUCTURED_TEXT_FILE);
		iphb->add_child(install_path);

		// status icon
		status_rect = memnew(TextureRect);
		status_rect->set_stretch_mode(TextureRect::STRETCH_KEEP_CENTERED);
		pphb->add_child(status_rect);

		browse = memnew(Button);
		browse->set_text(TTR("Browse"));
		browse->connect("pressed", callable_mp(this, &ProjectDialog::_browse_path));
		pphb->add_child(browse);

		// install status icon
		install_status_rect = memnew(TextureRect);
		install_status_rect->set_stretch_mode(TextureRect::STRETCH_KEEP_CENTERED);
		iphb->add_child(install_status_rect);

		install_browse = memnew(Button);
		install_browse->set_text(TTR("Browse"));
		install_browse->connect("pressed", callable_mp(this, &ProjectDialog::_browse_install_path));
		iphb->add_child(install_browse);

		msg = memnew(Label);
		msg->set_horizontal_alignment(HORIZONTAL_ALIGNMENT_CENTER);
		vb->add_child(msg);

		// rasterizer selection
		rasterizer_container = memnew(VBoxContainer);
		vb->add_child(rasterizer_container);
		l = memnew(Label);
		l->set_text(TTR("Renderer:"));
		rasterizer_container->add_child(l);
		Container *rshb = memnew(HBoxContainer);
		rasterizer_container->add_child(rshb);
		rasterizer_button_group.instantiate();

		Container *rvb = memnew(VBoxContainer);
		rvb->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		rshb->add_child(rvb);
		Button *rs_button = memnew(CheckBox);
		rs_button->set_button_group(rasterizer_button_group);
		rs_button->set_text(TTR("Vulkan Clustered"));
		rs_button->set_meta(SNAME("driver_name"), 0); // Vulkan backend "Forward Clustered"
		rs_button->set_pressed(true);
		rvb->add_child(rs_button);
		l = memnew(Label);
		l->set_text(
				String::utf8("•  ") + TTR("Supports desktop platforms only.") +
				String::utf8("\n•  ") + TTR("Advanced 3D graphics available.") +
				String::utf8("\n•  ") + TTR("Can scale to large complex scenes.") +
				String::utf8("\n•  ") + TTR("Slower rendering of simple scenes."));
		l->set_modulate(Color(1, 1, 1, 0.7));
		rvb->add_child(l);

		rshb->add_child(memnew(VSeparator));

		rvb = memnew(VBoxContainer);
		rvb->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		rshb->add_child(rvb);
		rs_button = memnew(CheckBox);
		rs_button->set_button_group(rasterizer_button_group);
		rs_button->set_text(TTR("Vulkan Mobile"));
		rs_button->set_meta(SNAME("driver_name"), 1); // Vulkan backend "Forward Mobile"
		rvb->add_child(rs_button);
		l = memnew(Label);
		l->set_text(
				String::utf8("•  ") + TTR("Supports desktop + mobile platforms.") +
				String::utf8("\n•  ") + TTR("Less advanced 3D graphics.") +
				String::utf8("\n•  ") + TTR("Less scalable for complex scenes.") +
				String::utf8("\n•  ") + TTR("Faster rendering of simple scenes."));
		l->set_modulate(Color(1, 1, 1, 0.7));
		rvb->add_child(l);

		l = memnew(Label);
		l->set_text(TTR("The renderer can be changed later, but scenes may need to be adjusted."));
		// Add some extra spacing to separate it from the list above and the buttons below.
		l->set_custom_minimum_size(Size2(0, 40) * EDSCALE);
		l->set_horizontal_alignment(HORIZONTAL_ALIGNMENT_CENTER);
		l->set_vertical_alignment(VERTICAL_ALIGNMENT_CENTER);
		l->set_modulate(Color(1, 1, 1, 0.7));
		rasterizer_container->add_child(l);

		default_files_container = memnew(HBoxContainer);
		vb->add_child(default_files_container);
		l = memnew(Label);
		l->set_text(TTR("Version Control Metadata:"));
		default_files_container->add_child(l);
		vcs_metadata_selection = memnew(OptionButton);
		vcs_metadata_selection->set_custom_minimum_size(Size2(100, 20));
		vcs_metadata_selection->add_item("None", (int)EditorVCSInterface::VCSMetadata::NONE);
		vcs_metadata_selection->add_item("Git", (int)EditorVCSInterface::VCSMetadata::GIT);
		vcs_metadata_selection->select((int)EditorVCSInterface::VCSMetadata::GIT);
		default_files_container->add_child(vcs_metadata_selection);
		Control *spacer = memnew(Control);
		spacer->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		default_files_container->add_child(spacer);

		fdialog = memnew(EditorFileDialog);
		fdialog->set_previews_enabled(false); //Crucial, otherwise the engine crashes.
		fdialog->set_access(EditorFileDialog::ACCESS_FILESYSTEM);
		fdialog_install = memnew(EditorFileDialog);
		fdialog_install->set_previews_enabled(false); //Crucial, otherwise the engine crashes.
		fdialog_install->set_access(EditorFileDialog::ACCESS_FILESYSTEM);
		add_child(fdialog);
		add_child(fdialog_install);

		project_name->connect("text_changed", callable_mp(this, &ProjectDialog::_text_changed));
		project_path->connect("text_changed", callable_mp(this, &ProjectDialog::_path_text_changed));
		install_path->connect("text_changed", callable_mp(this, &ProjectDialog::_path_text_changed));
		fdialog->connect("dir_selected", callable_mp(this, &ProjectDialog::_path_selected));
		fdialog->connect("file_selected", callable_mp(this, &ProjectDialog::_file_selected));
		fdialog_install->connect("dir_selected", callable_mp(this, &ProjectDialog::_install_path_selected));
		fdialog_install->connect("file_selected", callable_mp(this, &ProjectDialog::_install_path_selected));

		set_hide_on_ok(false);
		mode = MODE_NEW;

		dialog_error = memnew(AcceptDialog);
		add_child(dialog_error);
	}
};

class ProjectListItemControl : public HBoxContainer {
	GDCLASS(ProjectListItemControl, HBoxContainer)
public:
	TextureButton *favorite_button;
	TextureRect *icon;
	bool icon_needs_reload;
	bool hover;

	ProjectListItemControl() {
		favorite_button = nullptr;
		icon = nullptr;
		icon_needs_reload = true;
		hover = false;

		set_focus_mode(FocusMode::FOCUS_ALL);
	}

	void set_is_favorite(bool fav) {
		favorite_button->set_modulate(fav ? Color(1, 1, 1, 1) : Color(1, 1, 1, 0.2));
	}

	void _notification(int p_what) {
		switch (p_what) {
			case NOTIFICATION_MOUSE_ENTER: {
				hover = true;
				queue_redraw();
			} break;

			case NOTIFICATION_MOUSE_EXIT: {
				hover = false;
				queue_redraw();
			} break;

			case NOTIFICATION_DRAW: {
				if (hover) {
					draw_style_box(get_theme_stylebox(SNAME("hover"), SNAME("Tree")), Rect2(Point2(), get_size()));
				}
			} break;
		}
	}
};

class ProjectList : public ScrollContainer {
	GDCLASS(ProjectList, ScrollContainer)
public:
	static const char *SIGNAL_SELECTION_CHANGED;
	static const char *SIGNAL_PROJECT_ASK_OPEN;

	enum MenuOptions {
		GLOBAL_NEW_WINDOW,
		GLOBAL_OPEN_PROJECT
	};

	// Can often be passed by copy
	struct Item {
		String project_name;
		String description;
		String path;
		String icon;
		String main_scene;
		PackedStringArray unsupported_features;
		uint64_t last_edited = 0;
		bool favorite = false;
		bool grayed = false;
		bool missing = false;
		int version = 0;

		ProjectListItemControl *control = nullptr;

		Item() {}

		Item(const String &p_name,
				const String &p_description,
				const String &p_path,
				const String &p_icon,
				const String &p_main_scene,
				const PackedStringArray &p_unsupported_features,
				uint64_t p_last_edited,
				bool p_favorite,
				bool p_grayed,
				bool p_missing,
				int p_version) {
			project_name = p_name;
			description = p_description;
			path = p_path;
			icon = p_icon;
			main_scene = p_main_scene;
			unsupported_features = p_unsupported_features;
			last_edited = p_last_edited;
			favorite = p_favorite;
			grayed = p_grayed;
			missing = p_missing;
			version = p_version;
			control = nullptr;
		}

		_FORCE_INLINE_ bool operator==(const Item &l) const {
			return path == l.path;
		}
	};

	bool project_opening_initiated;

	ProjectList();
	~ProjectList();

	void _global_menu_new_window(const Variant &p_tag);
	void _global_menu_open_project(const Variant &p_tag);

	void update_dock_menu();
	void migrate_config();
	void load_projects();
	void set_search_term(String p_search_term);
	void set_order_option(int p_option);
	void sort_projects();
	int get_project_count() const;
	void select_project(int p_index);
	void select_first_visible_project();
	void erase_selected_projects(bool p_delete_project_contents);
	Vector<Item> get_selected_projects() const;
	const HashSet<String> &get_selected_project_keys() const;
	void ensure_project_visible(int p_index);
	int get_single_selected_index() const;
	bool is_any_project_missing() const;
	void erase_missing_projects();
	int refresh_project(const String &dir_path);
	void add_project(const String &dir_path, bool favorite);
	void save_config();
	void set_project_version(const String &p_project_path, int version);

private:
	static void _bind_methods();
	void _notification(int p_what);

	void _panel_draw(Node *p_hb);
	void _panel_input(const Ref<InputEvent> &p_ev, Node *p_hb);
	void _favorite_pressed(Node *p_hb);
	void _show_project(const String &p_path);

	void select_range(int p_begin, int p_end);
	void toggle_select(int p_index);
	void create_project_item_control(int p_index);
	void remove_project(int p_index, bool p_update_settings);
	void update_icons_async();
	void load_project_icon(int p_index);
	static Item load_project_data(const String &p_property_key, bool p_favorite);

	String _search_term;
	FilterOption _order_option;
	HashSet<String> _selected_project_paths;
	String _last_clicked; // Project key
	VBoxContainer *_scroll_children;
	int _icon_load_index;

	Vector<Item> _projects;

	ConfigFile _config;
	String _config_path;
};

struct ProjectListComparator {
	FilterOption order_option = FilterOption::EDIT_DATE;

	// operator<
	_FORCE_INLINE_ bool operator()(const ProjectList::Item &a, const ProjectList::Item &b) const {
		if (a.favorite && !b.favorite) {
			return true;
		}
		if (b.favorite && !a.favorite) {
			return false;
		}
		switch (order_option) {
			case PATH:
				return a.path < b.path;
			case EDIT_DATE:
				return a.last_edited > b.last_edited;
			default:
				return a.project_name < b.project_name;
		}
	}
};

ProjectList::ProjectList() {
	_order_option = FilterOption::EDIT_DATE;
	_scroll_children = memnew(VBoxContainer);
	_scroll_children->set_h_size_flags(Control::SIZE_EXPAND_FILL);
	add_child(_scroll_children);

	_icon_load_index = 0;
	project_opening_initiated = false;
	_config_path = EditorPaths::get_singleton()->get_data_dir().path_join("projects.cfg");
}

ProjectList::~ProjectList() {
}

void ProjectList::update_icons_async() {
	_icon_load_index = 0;
	set_process(true);
}

void ProjectList::_notification(int p_what) {
	switch (p_what) {
		case NOTIFICATION_PROCESS: {
			// Load icons as a coroutine to speed up launch when you have hundreds of projects
			if (_icon_load_index < _projects.size()) {
				Item &item = _projects.write[_icon_load_index];
				if (item.control->icon_needs_reload) {
					load_project_icon(_icon_load_index);
				}
				_icon_load_index++;

			} else {
				set_process(false);
			}
		} break;
	}
}

void ProjectList::load_project_icon(int p_index) {
	Item &item = _projects.write[p_index];

	Ref<Texture2D> default_icon = get_theme_icon(SNAME("DefaultProjectIcon"), SNAME("EditorIcons"));
	Ref<Texture2D> icon;
	if (!item.icon.is_empty()) {
		Ref<Image> img;
		img.instantiate();
		Error err = img->load(item.icon.replace_first("res://", item.path + "/"));
		if (err == OK) {
			img->resize(default_icon->get_width(), default_icon->get_height(), Image::INTERPOLATE_LANCZOS);
			icon = ImageTexture::create_from_image(img);
		}
	}
	if (icon.is_null()) {
		icon = default_icon;
	}

	// The default project icon is 128×128 to look crisp on hiDPI displays,
	// but we want the actual displayed size to be 64×64 on loDPI displays.
	item.control->icon->set_ignore_texture_size(true);
	item.control->icon->set_custom_minimum_size(Size2(64, 64) * EDSCALE);
	item.control->icon->set_stretch_mode(TextureRect::STRETCH_KEEP_ASPECT_CENTERED);

	item.control->icon->set_texture(icon);
	item.control->icon_needs_reload = false;
}

// Load project data from p_property_key and return it in a ProjectList::Item. p_favorite is passed directly into the Item.
ProjectList::Item ProjectList::load_project_data(const String &p_path, bool p_favorite) {
	String conf = p_path.path_join("project.godot");
	bool grayed = false;
	bool missing = false;

	Ref<ConfigFile> cf = memnew(ConfigFile);
	Error cf_err = cf->load(conf);

	int config_version = 0;
	String project_name = TTR("Unnamed Project");
	if (cf_err == OK) {
		String cf_project_name = static_cast<String>(cf->get_value("application", "config/name", ""));
		if (!cf_project_name.is_empty()) {
			project_name = cf_project_name.xml_unescape();
		}
		config_version = (int)cf->get_value("", "config_version", 0);
	}

	if (config_version > ProjectSettings::CONFIG_VERSION) {
		// Comes from an incompatible (more recent) Godot version, gray it out.
		grayed = true;
	}

	const String description = cf->get_value("application", "config/description", "");
	const String icon = cf->get_value("application", "config/icon", "");
	const String main_scene = cf->get_value("application", "run/main_scene", "");

	PackedStringArray project_features = cf->get_value("application", "config/features", PackedStringArray());
	PackedStringArray unsupported_features = ProjectSettings::get_unsupported_features(project_features);

	uint64_t last_edited = 0;
	if (FileAccess::exists(conf)) {
		// The modification date marks the date the project was last edited.
		// This is because the `project.godot` file will always be modified
		// when editing a project (but not when running it).
		last_edited = FileAccess::get_modified_time(conf);

		String fscache = p_path.path_join(".fscache");
		if (FileAccess::exists(fscache)) {
			uint64_t cache_modified = FileAccess::get_modified_time(fscache);
			if (cache_modified > last_edited) {
				last_edited = cache_modified;
			}
		}
	} else {
		grayed = true;
		missing = true;
		print_line("Project is missing: " + conf);
	}

	return Item(project_name, description, p_path, icon, main_scene, unsupported_features, last_edited, p_favorite, grayed, missing, config_version);
}

void ProjectList::migrate_config() {
	// Proposal #1637 moved the project list from editor settings to a separate config file
	// If the new config file doesn't exist, populate it from EditorSettings
	if (FileAccess::exists(_config_path)) {
		return;
	}
	print_line("Migrating legacy project list");

	List<PropertyInfo> properties;
	EditorSettings::get_singleton()->get_property_list(&properties);

	for (const PropertyInfo &E : properties) {
		// This is actually something like "projects/C:::Documents::Godot::Projects::MyGame"
		String property_key = E.name;
		if (!property_key.begins_with("projects/")) {
			continue;
		}

		String path = EditorSettings::get_singleton()->get(property_key);
		String favoriteKey = "favorite_projects/" + property_key.get_slice("/", 1);
		bool favorite = EditorSettings::get_singleton()->has_setting(favoriteKey);
		add_project(path, favorite);
		if (favorite) {
			EditorSettings::get_singleton()->erase(favoriteKey);
		}
		EditorSettings::get_singleton()->erase(property_key);
	}

	save_config();
}

void ProjectList::load_projects() {
	// This is a full, hard reload of the list. Don't call this unless really required, it's expensive.
	// If you have 150 projects, it may read through 150 files on your disk at once + load 150 icons.

	// Clear whole list
	for (int i = 0; i < _projects.size(); ++i) {
		Item &project = _projects.write[i];
		CRASH_COND(project.control == nullptr);
		memdelete(project.control); // Why not queue_free()?
	}
	_projects.clear();
	_last_clicked = "";
	_selected_project_paths.clear();

	List<String> sections;
	_config.load(_config_path);
	_config.get_sections(&sections);

	for (const String &path : sections) {
		bool favorite = _config.get_value(path, "favorite", false);
		_projects.push_back(load_project_data(path, favorite));
	}

	// Create controls
	for (int i = 0; i < _projects.size(); ++i) {
		create_project_item_control(i);
	}

	set_v_scroll(0);

	update_icons_async();

	update_dock_menu();
}

void ProjectList::update_dock_menu() {
	if (!DisplayServer::get_singleton()->has_feature(DisplayServer::FEATURE_GLOBAL_MENU)) {
		return;
	}
	DisplayServer::get_singleton()->global_menu_clear("_dock");

	int favs_added = 0;
	int total_added = 0;
	for (int i = 0; i < _projects.size(); ++i) {
		if (!_projects[i].grayed && !_projects[i].missing) {
			if (_projects[i].favorite) {
				favs_added++;
			} else {
				if (favs_added != 0) {
					DisplayServer::get_singleton()->global_menu_add_separator("_dock");
				}
				favs_added = 0;
			}
			DisplayServer::get_singleton()->global_menu_add_item("_dock", _projects[i].project_name + " ( " + _projects[i].path + " )", callable_mp(this, &ProjectList::_global_menu_open_project), Callable(), i);
			total_added++;
		}
	}
	if (total_added != 0) {
		DisplayServer::get_singleton()->global_menu_add_separator("_dock");
	}
	DisplayServer::get_singleton()->global_menu_add_item("_dock", TTR("New Window"), callable_mp(this, &ProjectList::_global_menu_new_window));
}

void ProjectList::_global_menu_new_window(const Variant &p_tag) {
	List<String> args;
	args.push_back("-p");
	OS::get_singleton()->create_instance(args);
}

void ProjectList::_global_menu_open_project(const Variant &p_tag) {
	int idx = (int)p_tag;

	if (idx >= 0 && idx < _projects.size()) {
		String conf = _projects[idx].path.path_join("project.godot");
		List<String> args;
		args.push_back(conf);
		OS::get_singleton()->create_instance(args);
	}
}

void ProjectList::create_project_item_control(int p_index) {
	// Will be added last in the list, so make sure indexes match
	ERR_FAIL_COND(p_index != _scroll_children->get_child_count());

	Item &item = _projects.write[p_index];
	ERR_FAIL_COND(item.control != nullptr); // Already created

	Ref<Texture2D> favorite_icon = get_theme_icon(SNAME("Favorites"), SNAME("EditorIcons"));
	Color font_color = get_theme_color(SNAME("font_color"), SNAME("Tree"));

	ProjectListItemControl *hb = memnew(ProjectListItemControl);
	hb->connect("draw", callable_mp(this, &ProjectList::_panel_draw).bind(hb));
	hb->connect("gui_input", callable_mp(this, &ProjectList::_panel_input).bind(hb));
	hb->add_theme_constant_override("separation", 10 * EDSCALE);
	hb->set_tooltip_text(item.description);

	VBoxContainer *favorite_box = memnew(VBoxContainer);
	favorite_box->set_name("FavoriteBox");
	TextureButton *favorite = memnew(TextureButton);
	favorite->set_name("FavoriteButton");
	favorite->set_normal_texture(favorite_icon);
	// This makes the project's "hover" style display correctly when hovering the favorite icon.
	favorite->set_mouse_filter(MOUSE_FILTER_PASS);
	favorite->connect("pressed", callable_mp(this, &ProjectList::_favorite_pressed).bind(hb));
	favorite_box->add_child(favorite);
	favorite_box->set_alignment(BoxContainer::ALIGNMENT_CENTER);
	hb->add_child(favorite_box);
	hb->favorite_button = favorite;
	hb->set_is_favorite(item.favorite);

	TextureRect *tf = memnew(TextureRect);
	// The project icon may not be loaded by the time the control is displayed,
	// so use a loading placeholder.
	tf->set_texture(get_theme_icon(SNAME("ProjectIconLoading"), SNAME("EditorIcons")));
	tf->set_v_size_flags(SIZE_SHRINK_CENTER);
	if (item.missing) {
		tf->set_modulate(Color(1, 1, 1, 0.5));
	}
	hb->add_child(tf);
	hb->icon = tf;

	VBoxContainer *vb = memnew(VBoxContainer);
	if (item.grayed) {
		vb->set_modulate(Color(1, 1, 1, 0.5));
	}
	vb->set_h_size_flags(Control::SIZE_EXPAND_FILL);
	hb->add_child(vb);
	Control *ec = memnew(Control);
	ec->set_custom_minimum_size(Size2(0, 1));
	ec->set_mouse_filter(MOUSE_FILTER_PASS);
	vb->add_child(ec);

	{ // Top half, title and unsupported features labels.
		HBoxContainer *title_hb = memnew(HBoxContainer);
		vb->add_child(title_hb);

		Label *title = memnew(Label(!item.missing ? item.project_name : TTR("Missing Project")));
		title->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		title->add_theme_font_override("font", get_theme_font(SNAME("title"), SNAME("EditorFonts")));
		title->add_theme_font_size_override("font_size", get_theme_font_size(SNAME("title_size"), SNAME("EditorFonts")));
		title->add_theme_color_override("font_color", font_color);
		title->set_clip_text(true);
		title_hb->add_child(title);

		String unsupported_features_str = String(", ").join(item.unsupported_features);
		int length = unsupported_features_str.length();
		if (length > 0) {
			Label *unsupported_label = memnew(Label(unsupported_features_str));
			unsupported_label->set_custom_minimum_size(Size2(length * 15, 10) * EDSCALE);
			unsupported_label->add_theme_font_override("font", get_theme_font(SNAME("title"), SNAME("EditorFonts")));
			unsupported_label->add_theme_color_override("font_color", get_theme_color(SNAME("warning_color"), SNAME("Editor")));
			unsupported_label->set_clip_text(true);
			unsupported_label->set_horizontal_alignment(HORIZONTAL_ALIGNMENT_RIGHT);
			title_hb->add_child(unsupported_label);
			Control *spacer = memnew(Control());
			spacer->set_custom_minimum_size(Size2(10, 10));
			title_hb->add_child(spacer);
		}
	}

	{ // Bottom half, containing the path and view folder button.
		HBoxContainer *path_hb = memnew(HBoxContainer);
		path_hb->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		vb->add_child(path_hb);

		Button *show = memnew(Button);
		// Display a folder icon if the project directory can be opened, or a "broken file" icon if it can't.
		show->set_icon(get_theme_icon(!item.missing ? SNAME("Load") : SNAME("FileBroken"), SNAME("EditorIcons")));
		show->set_flat(true);
		if (!item.grayed) {
			// Don't make the icon less prominent if the parent is already grayed out.
			show->set_modulate(Color(1, 1, 1, 0.5));
		}
		path_hb->add_child(show);

		if (!item.missing) {
			show->connect("pressed", callable_mp(this, &ProjectList::_show_project).bind(item.path));
			show->set_tooltip_text(TTR("Show in File Manager"));
		} else {
			show->set_tooltip_text(TTR("Error: Project is missing on the filesystem."));
		}

		Label *fpath = memnew(Label(item.path));
		fpath->set_structured_text_bidi_override(TextServer::STRUCTURED_TEXT_FILE);
		path_hb->add_child(fpath);
		fpath->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		fpath->set_modulate(Color(1, 1, 1, 0.5));
		fpath->add_theme_color_override("font_color", font_color);
		fpath->set_clip_text(true);
	}

	_scroll_children->add_child(hb);
	item.control = hb;
}

void ProjectList::set_search_term(String p_search_term) {
	_search_term = p_search_term;
}

void ProjectList::set_order_option(int p_option) {
	FilterOption selected = (FilterOption)p_option;
	EditorSettings::get_singleton()->set("project_manager/sorting_order", p_option);
	EditorSettings::get_singleton()->save();
	_order_option = selected;

	sort_projects();
}

void ProjectList::sort_projects() {
	SortArray<Item, ProjectListComparator> sorter;
	sorter.compare.order_option = _order_option;
	sorter.sort(_projects.ptrw(), _projects.size());

	for (int i = 0; i < _projects.size(); ++i) {
		Item &item = _projects.write[i];

		bool visible = true;
		if (!_search_term.is_empty()) {
			String search_path;
			if (_search_term.contains("/")) {
				// Search path will match the whole path
				search_path = item.path;
			} else {
				// Search path will only match the last path component to make searching more strict
				search_path = item.path.get_file();
			}

			// When searching, display projects whose name or path contain the search term
			visible = item.project_name.findn(_search_term) != -1 || search_path.findn(_search_term) != -1;
		}

		item.control->set_visible(visible);
	}

	for (int i = 0; i < _projects.size(); ++i) {
		Item &item = _projects.write[i];
		item.control->get_parent()->move_child(item.control, i);
	}

	// Rewind the coroutine because order of projects changed
	update_icons_async();

	update_dock_menu();
}

const HashSet<String> &ProjectList::get_selected_project_keys() const {
	// Faster if that's all you need
	return _selected_project_paths;
}

Vector<ProjectList::Item> ProjectList::get_selected_projects() const {
	Vector<Item> items;
	if (_selected_project_paths.size() == 0) {
		return items;
	}
	items.resize(_selected_project_paths.size());
	int j = 0;
	for (int i = 0; i < _projects.size(); ++i) {
		const Item &item = _projects[i];
		if (_selected_project_paths.has(item.path)) {
			items.write[j++] = item;
		}
	}
	ERR_FAIL_COND_V(j != items.size(), items);
	return items;
}

void ProjectList::ensure_project_visible(int p_index) {
	const Item &item = _projects[p_index];
	ensure_control_visible(item.control);
}

int ProjectList::get_single_selected_index() const {
	if (_selected_project_paths.size() == 0) {
		// Default selection
		return 0;
	}
	String key;
	if (_selected_project_paths.size() == 1) {
		// Only one selected
		key = *_selected_project_paths.begin();
	} else {
		// Multiple selected, consider the last clicked one as "main"
		key = _last_clicked;
	}
	for (int i = 0; i < _projects.size(); ++i) {
		if (_projects[i].path == key) {
			return i;
		}
	}
	return 0;
}

void ProjectList::remove_project(int p_index, bool p_update_config) {
	const Item item = _projects[p_index]; // Take a copy

	_selected_project_paths.erase(item.path);

	if (_last_clicked == item.path) {
		_last_clicked = "";
	}

	memdelete(item.control);
	_projects.remove_at(p_index);

	if (p_update_config) {
		_config.erase_section(item.path);
		// Not actually saving the file, in case you are doing more changes to settings
	}

	update_dock_menu();
}

bool ProjectList::is_any_project_missing() const {
	for (int i = 0; i < _projects.size(); ++i) {
		if (_projects[i].missing) {
			return true;
		}
	}
	return false;
}

void ProjectList::erase_missing_projects() {
	if (_projects.is_empty()) {
		return;
	}

	int deleted_count = 0;
	int remaining_count = 0;

	for (int i = 0; i < _projects.size(); ++i) {
		const Item &item = _projects[i];

		if (item.missing) {
			remove_project(i, true);
			--i;
			++deleted_count;

		} else {
			++remaining_count;
		}
	}

	print_line("Removed " + itos(deleted_count) + " projects from the list, remaining " + itos(remaining_count) + " projects");
	save_config();
}

int ProjectList::refresh_project(const String &dir_path) {
	// Reloads information about a specific project.
	// If it wasn't loaded and should be in the list, it is added (i.e new project).
	// If it isn't in the list anymore, it is removed.
	// If it is in the list but doesn't exist anymore, it is marked as missing.

	bool should_be_in_list = _config.has_section(dir_path);
	bool is_favourite = _config.get_value(dir_path, "favorite", false);

	bool was_selected = _selected_project_paths.has(dir_path);

	// Remove item in any case
	for (int i = 0; i < _projects.size(); ++i) {
		const Item &existing_item = _projects[i];
		if (existing_item.path == dir_path) {
			remove_project(i, false);
			break;
		}
	}

	int index = -1;
	if (should_be_in_list) {
		// Recreate it with updated info

		Item item = load_project_data(dir_path, is_favourite);

		_projects.push_back(item);
		create_project_item_control(_projects.size() - 1);

		sort_projects();

		for (int i = 0; i < _projects.size(); ++i) {
			if (_projects[i].path == dir_path) {
				if (was_selected) {
					select_project(i);
					ensure_project_visible(i);
				}
				load_project_icon(i);

				index = i;
				break;
			}
		}
	}

	return index;
}

void ProjectList::add_project(const String &dir_path, bool favorite) {
	if (!_config.has_section(dir_path)) {
		_config.set_value(dir_path, "favorite", favorite);
	}
}

void ProjectList::save_config() {
	_config.save(_config_path);
}

void ProjectList::set_project_version(const String &p_project_path, int p_version) {
	for (ProjectList::Item &E : _projects) {
		if (E.path == p_project_path) {
			E.version = p_version;
			break;
		}
	}
}

int ProjectList::get_project_count() const {
	return _projects.size();
}

void ProjectList::select_project(int p_index) {
	Vector<Item> previous_selected_items = get_selected_projects();
	_selected_project_paths.clear();

	for (int i = 0; i < previous_selected_items.size(); ++i) {
		previous_selected_items[i].control->queue_redraw();
	}

	toggle_select(p_index);
}

void ProjectList::select_first_visible_project() {
	bool found = false;

	for (int i = 0; i < _projects.size(); i++) {
		if (_projects[i].control->is_visible()) {
			select_project(i);
			found = true;
			break;
		}
	}

	if (!found) {
		// Deselect all projects if there are no visible projects in the list.
		_selected_project_paths.clear();
	}
}

inline void sort(int &a, int &b) {
	if (a > b) {
		int temp = a;
		a = b;
		b = temp;
	}
}

void ProjectList::select_range(int p_begin, int p_end) {
	sort(p_begin, p_end);
	select_project(p_begin);
	for (int i = p_begin + 1; i <= p_end; ++i) {
		toggle_select(i);
	}
}

void ProjectList::toggle_select(int p_index) {
	Item &item = _projects.write[p_index];
	if (_selected_project_paths.has(item.path)) {
		_selected_project_paths.erase(item.path);
	} else {
		_selected_project_paths.insert(item.path);
	}
	item.control->queue_redraw();
}

void ProjectList::erase_selected_projects(bool p_delete_project_contents) {
	if (_selected_project_paths.size() == 0) {
		return;
	}

	for (int i = 0; i < _projects.size(); ++i) {
		Item &item = _projects.write[i];
		if (_selected_project_paths.has(item.path) && item.control->is_visible()) {
			_config.erase_section(item.path);

			if (p_delete_project_contents) {
				OS::get_singleton()->move_to_trash(item.path);
			}

			memdelete(item.control);
			_projects.remove_at(i);
			--i;
		}
	}

	save_config();
	_selected_project_paths.clear();
	_last_clicked = "";

	update_dock_menu();
}

// Draws selected project highlight
void ProjectList::_panel_draw(Node *p_hb) {
	Control *hb = Object::cast_to<Control>(p_hb);

	if (is_layout_rtl() && get_v_scroll_bar()->is_visible_in_tree()) {
		hb->draw_line(Point2(get_v_scroll_bar()->get_minimum_size().x, hb->get_size().y + 1), Point2(hb->get_size().x, hb->get_size().y + 1), get_theme_color(SNAME("guide_color"), SNAME("Tree")));
	} else {
		hb->draw_line(Point2(0, hb->get_size().y + 1), Point2(hb->get_size().x, hb->get_size().y + 1), get_theme_color(SNAME("guide_color"), SNAME("Tree")));
	}

	String key = _projects[p_hb->get_index()].path;

	if (_selected_project_paths.has(key)) {
		hb->draw_style_box(get_theme_stylebox(SNAME("selected"), SNAME("Tree")), Rect2(Point2(), hb->get_size()));
	}
}

// Input for each item in the list
void ProjectList::_panel_input(const Ref<InputEvent> &p_ev, Node *p_hb) {
	Ref<InputEventMouseButton> mb = p_ev;
	int clicked_index = p_hb->get_index();
	const Item &clicked_project = _projects[clicked_index];

	if (mb.is_valid() && mb->is_pressed() && mb->get_button_index() == MouseButton::LEFT) {
		if (mb->is_shift_pressed() && _selected_project_paths.size() > 0 && !_last_clicked.is_empty() && clicked_project.path != _last_clicked) {
			int anchor_index = -1;
			for (int i = 0; i < _projects.size(); ++i) {
				const Item &p = _projects[i];
				if (p.path == _last_clicked) {
					anchor_index = p.control->get_index();
					break;
				}
			}
			CRASH_COND(anchor_index == -1);
			select_range(anchor_index, clicked_index);

		} else if (mb->is_ctrl_pressed()) {
			toggle_select(clicked_index);

		} else {
			_last_clicked = clicked_project.path;
			select_project(clicked_index);
		}

		emit_signal(SNAME(SIGNAL_SELECTION_CHANGED));

		// Do not allow opening a project more than once using a single project manager instance.
		// Opening the same project in several editor instances at once can lead to various issues.
		if (!mb->is_ctrl_pressed() && mb->is_double_click() && !project_opening_initiated) {
			emit_signal(SNAME(SIGNAL_PROJECT_ASK_OPEN));
		}
	}
}

void ProjectList::_favorite_pressed(Node *p_hb) {
	ProjectListItemControl *control = Object::cast_to<ProjectListItemControl>(p_hb);

	int index = control->get_index();
	Item item = _projects.write[index]; // Take copy

	item.favorite = !item.favorite;

	_config.set_value(item.path, "favorite", item.favorite);
	save_config();

	_projects.write[index] = item;

	control->set_is_favorite(item.favorite);

	sort_projects();

	if (item.favorite) {
		for (int i = 0; i < _projects.size(); ++i) {
			if (_projects[i].path == item.path) {
				ensure_project_visible(i);
				break;
			}
		}
	}

	update_dock_menu();
}

void ProjectList::_show_project(const String &p_path) {
	OS::get_singleton()->shell_open(String("file://") + p_path);
}

const char *ProjectList::SIGNAL_SELECTION_CHANGED = "selection_changed";
const char *ProjectList::SIGNAL_PROJECT_ASK_OPEN = "project_ask_open";

void ProjectList::_bind_methods() {
	ADD_SIGNAL(MethodInfo(SIGNAL_SELECTION_CHANGED));
	ADD_SIGNAL(MethodInfo(SIGNAL_PROJECT_ASK_OPEN));
}

ProjectManager *ProjectManager::singleton = nullptr;

void ProjectManager::_notification(int p_what) {
	switch (p_what) {
		case NOTIFICATION_TRANSLATION_CHANGED:
		case NOTIFICATION_LAYOUT_DIRECTION_CHANGED: {
			settings_hb->set_anchors_and_offsets_preset(Control::PRESET_TOP_RIGHT);
			queue_redraw();
		} break;

		case NOTIFICATION_ENTER_TREE: {
			search_box->set_right_icon(get_theme_icon(SNAME("Search"), SNAME("EditorIcons")));
			search_box->set_clear_button_enabled(true);

			create_btn->set_icon(get_theme_icon(SNAME("Add"), SNAME("EditorIcons")));
			import_btn->set_icon(get_theme_icon(SNAME("Load"), SNAME("EditorIcons")));
			scan_btn->set_icon(get_theme_icon(SNAME("Search"), SNAME("EditorIcons")));
			open_btn->set_icon(get_theme_icon(SNAME("Edit"), SNAME("EditorIcons")));
			run_btn->set_icon(get_theme_icon(SNAME("Play"), SNAME("EditorIcons")));
			rename_btn->set_icon(get_theme_icon(SNAME("Rename"), SNAME("EditorIcons")));
			erase_btn->set_icon(get_theme_icon(SNAME("Remove"), SNAME("EditorIcons")));
			erase_missing_btn->set_icon(get_theme_icon(SNAME("Clear"), SNAME("EditorIcons")));

			Engine::get_singleton()->set_editor_hint(false);
		} break;

		case NOTIFICATION_RESIZED: {
			if (open_templates && open_templates->is_visible()) {
				open_templates->popup_centered();
			}
			if (asset_library) {
				real_t size = get_size().x / EDSCALE;
				// Adjust names of tabs to fit the new size.
				if (size < 650) {
					local_projects_hb->set_name(TTR("Local"));
					asset_library->set_name(TTR("Asset Library"));
				} else {
					local_projects_hb->set_name(TTR("Local Projects"));
					asset_library->set_name(TTR("Asset Library Projects"));
				}
			}
		} break;

		case NOTIFICATION_READY: {
			int default_sorting = (int)EditorSettings::get_singleton()->get("project_manager/sorting_order");
			filter_option->select(default_sorting);
			_project_list->set_order_option(default_sorting);

#ifndef ANDROID_ENABLED
			if (_project_list->get_project_count() >= 1) {
				// Focus on the search box immediately to allow the user
				// to search without having to reach for their mouse
				search_box->grab_focus();
			}
#endif

			if (asset_library) {
				// Removes extra border margins.
				asset_library->add_theme_style_override("panel", memnew(StyleBoxEmpty));
				// Suggest browsing asset library to get templates/demos.
				if (open_templates && _project_list->get_project_count() == 0) {
					open_templates->popup_centered();
				}
			}
		} break;

		case NOTIFICATION_VISIBILITY_CHANGED: {
			set_process_shortcut_input(is_visible_in_tree());
		} break;

		case NOTIFICATION_WM_CLOSE_REQUEST: {
			_dim_window();
		} break;

		case NOTIFICATION_WM_ABOUT: {
			_show_about();
		} break;
	}
}

Ref<Texture2D> ProjectManager::_file_dialog_get_icon(const String &p_path) {
	return singleton->icon_type_cache["ObjectHR"];
}

void ProjectManager::_build_icon_type_cache(Ref<Theme> p_theme) {
	List<StringName> tl;
	p_theme->get_icon_list(SNAME("EditorIcons"), &tl);
	for (List<StringName>::Element *E = tl.front(); E; E = E->next()) {
		if (!ClassDB::class_exists(E->get())) {
			continue;
		}
		icon_type_cache[E->get()] = p_theme->get_icon(E->get(), SNAME("EditorIcons"));
	}
}

void ProjectManager::_dim_window() {
	// This method must be called before calling `get_tree()->quit()`.
	// Otherwise, its effect won't be visible

	// Dim the project manager window while it's quitting to make it clearer that it's busy.
	// No transition is applied, as the effect needs to be visible immediately
	float c = 0.5f;
	Color dim_color = Color(c, c, c);
	set_modulate(dim_color);
}

void ProjectManager::_update_project_buttons() {
	Vector<ProjectList::Item> selected_projects = _project_list->get_selected_projects();
	bool empty_selection = selected_projects.is_empty();

	bool is_missing_project_selected = false;
	for (int i = 0; i < selected_projects.size(); ++i) {
		if (selected_projects[i].missing) {
			is_missing_project_selected = true;
			break;
		}
	}

	erase_btn->set_disabled(empty_selection);
	open_btn->set_disabled(empty_selection || is_missing_project_selected);
	rename_btn->set_disabled(empty_selection || is_missing_project_selected);
	run_btn->set_disabled(empty_selection || is_missing_project_selected);

	erase_missing_btn->set_disabled(!_project_list->is_any_project_missing());
}

void ProjectManager::shortcut_input(const Ref<InputEvent> &p_ev) {
	ERR_FAIL_COND(p_ev.is_null());

	Ref<InputEventKey> k = p_ev;

	if (k.is_valid()) {
		if (!k->is_pressed()) {
			return;
		}

		// Pressing Command + Q quits the Project Manager
		// This is handled by the platform implementation on macOS,
		// so only define the shortcut on other platforms
#ifndef MACOS_ENABLED
		if (k->get_keycode_with_modifiers() == (KeyModifierMask::META | Key::Q)) {
			_dim_window();
			get_tree()->quit();
		}
#endif

		if (tabs->get_current_tab() != 0) {
			return;
		}

		bool keycode_handled = true;

		switch (k->get_keycode()) {
			case Key::ENTER: {
				_open_selected_projects_ask();
			} break;
			case Key::HOME: {
				if (_project_list->get_project_count() > 0) {
					_project_list->select_project(0);
					_update_project_buttons();
				}

			} break;
			case Key::END: {
				if (_project_list->get_project_count() > 0) {
					_project_list->select_project(_project_list->get_project_count() - 1);
					_update_project_buttons();
				}

			} break;
			case Key::UP: {
				if (k->is_shift_pressed()) {
					break;
				}

				int index = _project_list->get_single_selected_index();
				if (index > 0) {
					_project_list->select_project(index - 1);
					_project_list->ensure_project_visible(index - 1);
					_update_project_buttons();
				}

				break;
			}
			case Key::DOWN: {
				if (k->is_shift_pressed()) {
					break;
				}

				int index = _project_list->get_single_selected_index();
				if (index + 1 < _project_list->get_project_count()) {
					_project_list->select_project(index + 1);
					_project_list->ensure_project_visible(index + 1);
					_update_project_buttons();
				}

			} break;
			case Key::F: {
				if (k->is_command_or_control_pressed()) {
					this->search_box->grab_focus();
				} else {
					keycode_handled = false;
				}
			} break;
			default: {
				keycode_handled = false;
			} break;
		}

		if (keycode_handled) {
			accept_event();
		}
	}
}

void ProjectManager::_load_recent_projects() {
	_project_list->set_search_term(search_box->get_text().strip_edges());
	_project_list->load_projects();

	_update_project_buttons();

	tabs->set_current_tab(0);
}

void ProjectManager::_on_projects_updated() {
	Vector<ProjectList::Item> selected_projects = _project_list->get_selected_projects();
	int index = 0;
	for (int i = 0; i < selected_projects.size(); ++i) {
		index = _project_list->refresh_project(selected_projects[i].path);
	}
	if (index != -1) {
		_project_list->ensure_project_visible(index);
	}

	_project_list->update_dock_menu();
}

void ProjectManager::_on_project_created(const String &dir) {
	_project_list->add_project(dir, false);
	_project_list->save_config();
	search_box->clear();
	int i = _project_list->refresh_project(dir);
	_project_list->select_project(i);
	_project_list->ensure_project_visible(i);
	_open_selected_projects_ask();

	_project_list->update_dock_menu();
}

void ProjectManager::_confirm_update_settings() {
	_open_selected_projects();
}

void ProjectManager::_open_selected_projects() {
	// Show loading text to tell the user that the project manager is busy loading.
	// This is especially important for the Web project manager.
	loading_label->show();

	const HashSet<String> &selected_list = _project_list->get_selected_project_keys();

	for (const String &path : selected_list) {
		String conf = path.path_join("project.godot");

		if (!FileAccess::exists(conf)) {
			dialog_error->set_text(vformat(TTR("Can't open project at '%s'."), path));
			dialog_error->popup_centered();
			return;
		}

		print_line("Editing project: " + path);

		List<String> args;

		for (const String &a : Main::get_forwardable_cli_arguments(Main::CLI_SCOPE_TOOL)) {
			args.push_back(a);
		}

		args.push_back("--path");
		args.push_back(path);

		args.push_back("--editor");

		Error err = OS::get_singleton()->create_instance(args);
		ERR_FAIL_COND(err);
	}

	_project_list->project_opening_initiated = true;

	_dim_window();
	get_tree()->quit();
}

void ProjectManager::_open_selected_projects_ask() {
	const HashSet<String> &selected_list = _project_list->get_selected_project_keys();

	if (selected_list.size() < 1) {
		return;
	}

	if (selected_list.size() > 1) {
		multi_open_ask->set_text(TTR("Are you sure to open more than one project?"));
		multi_open_ask->popup_centered();
		return;
	}

	ProjectList::Item project = _project_list->get_selected_projects()[0];
	if (project.missing) {
		return;
	}

	// Update the project settings or don't open
	const String conf = project.path.path_join("project.godot");
	const int config_version = project.version;
	PackedStringArray unsupported_features = project.unsupported_features;

	Label *ask_update_label = ask_update_settings->get_label();
	ask_update_label->set_horizontal_alignment(HORIZONTAL_ALIGNMENT_LEFT); // Reset in case of previous center align.
	full_convert_button->hide();

	ask_update_settings->get_ok_button()->set_text("OK");

	// Check if the config_version property was empty or 0
	if (config_version == 0) {
		ask_update_settings->set_text(vformat(TTR("The following project settings file does not specify the version of Godot through which it was created.\n\n%s\n\nIf you proceed with opening it, it will be converted to Godot's current configuration file format.\nWarning: You won't be able to open the project with previous versions of the engine anymore."), conf));
		ask_update_settings->popup_centered();
		return;
	}
	// Check if we need to convert project settings from an earlier engine version
	if (config_version < ProjectSettings::CONFIG_VERSION) {
		if (config_version == GODOT4_CONFIG_VERSION - 1 && ProjectSettings::CONFIG_VERSION == GODOT4_CONFIG_VERSION) { // Conversion from Godot 3 to 4.
			full_convert_button->show();
		}
		ask_update_settings->set_text(vformat(TTR("The following project settings file was generated by an older engine version, and needs to be converted for this version:\n\n%s\n\nDo you want to convert it? You can also convert the entire project (recommended if you are upgrading).\nWarning: You won't be able to open the project with previous versions of the engine anymore."), conf));
		ask_update_settings->get_ok_button()->set_text(TTR("Convert project.godot"));
		ask_update_settings->popup_centered();
		return;
	}
	// Check if the file was generated by a newer, incompatible engine version
	if (config_version > ProjectSettings::CONFIG_VERSION) {
		dialog_error->set_text(vformat(TTR("Can't open project at '%s'.") + "\n" + TTR("The project settings were created by a newer engine version, whose settings are not compatible with this version."), project.path));
		dialog_error->popup_centered();
		return;
	}
	// Check if the project is using features not supported by this build of Godot.
	if (!unsupported_features.is_empty()) {
		String warning_message = "";
		for (int i = 0; i < unsupported_features.size(); i++) {
			String feature = unsupported_features[i];
			if (feature == "Double Precision") {
				warning_message += TTR("Warning: This project uses double precision floats, but this version of\nGodot uses single precision floats. Opening this project may cause data loss.\n\n");
				unsupported_features.remove_at(i);
				i--;
			} else if (feature == "C#") {
				warning_message += TTR("Warning: This project uses C#, but this build of Godot does not have\nthe Mono module. If you proceed you will not be able to use any C# scripts.\n\n");
				unsupported_features.remove_at(i);
				i--;
			} else if (feature.substr(0, 3).is_numeric()) {
				warning_message += vformat(TTR("Warning: This project was built in Godot %s.\nOpening will upgrade or downgrade the project to Godot %s.\n\n"), Variant(feature), Variant(VERSION_BRANCH));
				unsupported_features.remove_at(i);
				i--;
			}
		}
		if (!unsupported_features.is_empty()) {
			String unsupported_features_str = String(", ").join(unsupported_features);
			warning_message += vformat(TTR("Warning: This project uses the following features not supported by this build of Godot:\n\n%s\n\n"), unsupported_features_str);
		}
		warning_message += TTR("Open anyway? Project will be modified.");
		ask_update_label->set_horizontal_alignment(HORIZONTAL_ALIGNMENT_CENTER);
		ask_update_settings->set_text(warning_message);
		ask_update_settings->popup_centered();
		return;
	}

	// Open if the project is up-to-date
	_open_selected_projects();
}

void ProjectManager::_perform_full_project_conversion() {
	Vector<ProjectList::Item> selected_list = _project_list->get_selected_projects();
	if (selected_list.is_empty()) {
		return;
	}

	const String &path = selected_list[0].path;

	print_line("Converting project: " + path);

	Ref<ConfigFile> cf;
	cf.instantiate();
	cf->load(path.path_join("project.godot"));
	cf->set_value("", "config_version", GODOT4_CONFIG_VERSION);
	cf->save(path.path_join("project.godot"));
	_project_list->set_project_version(path, GODOT4_CONFIG_VERSION);

	List<String> args;
	args.push_back("--path");
	args.push_back(path);
	args.push_back("--convert-3to4");

	Error err = OS::get_singleton()->create_instance(args);
	ERR_FAIL_COND(err);
}

void ProjectManager::_run_project_confirm() {
	Vector<ProjectList::Item> selected_list = _project_list->get_selected_projects();

	for (int i = 0; i < selected_list.size(); ++i) {
		const String &selected_main = selected_list[i].main_scene;
		if (selected_main.is_empty()) {
			run_error_diag->set_text(TTR("Can't run project: no main scene defined.\nPlease edit the project and set the main scene in the Project Settings under the \"Application\" category."));
			run_error_diag->popup_centered();
			continue;
		}

		const String &path = selected_list[i].path;

		// `.substr(6)` on `ProjectSettings::get_singleton()->get_imported_files_path()` strips away the leading "res://".
		if (!DirAccess::exists(path.path_join(ProjectSettings::get_singleton()->get_imported_files_path().substr(6)))) {
			run_error_diag->set_text(TTR("Can't run project: Assets need to be imported.\nPlease edit the project to trigger the initial import."));
			run_error_diag->popup_centered();
			continue;
		}

		print_line("Running project: " + path);

		List<String> args;

		for (const String &a : Main::get_forwardable_cli_arguments(Main::CLI_SCOPE_PROJECT)) {
			args.push_back(a);
		}

		args.push_back("--path");
		args.push_back(path);

		Error err = OS::get_singleton()->create_instance(args);
		ERR_FAIL_COND(err);
	}
}

void ProjectManager::_run_project() {
	const HashSet<String> &selected_list = _project_list->get_selected_project_keys();

	if (selected_list.size() < 1) {
		return;
	}

	if (selected_list.size() > 1) {
		multi_run_ask->set_text(vformat(TTR("Are you sure to run %d projects at once?"), selected_list.size()));
		multi_run_ask->popup_centered();
	} else {
		_run_project_confirm();
	}
}

void ProjectManager::_scan_dir(const String &path) {
	Ref<DirAccess> da = DirAccess::create(DirAccess::ACCESS_FILESYSTEM);
	Error error = da->change_dir(path);
	ERR_FAIL_COND_MSG(error != OK, "Could not scan directory at: " + path);
	da->list_dir_begin();
	String n = da->get_next();
	while (!n.is_empty()) {
		if (da->current_is_dir() && !n.begins_with(".")) {
			_scan_dir(da->get_current_dir().path_join(n));
		} else if (n == "project.godot") {
			_project_list->add_project(da->get_current_dir(), false);
		}
		n = da->get_next();
	}
	da->list_dir_end();
}
void ProjectManager::_scan_begin(const String &p_base) {
	print_line("Scanning projects at: " + p_base);
	_scan_dir(p_base);
	_project_list->save_config();
	_load_recent_projects();
}

void ProjectManager::_scan_projects() {
	scan_dir->popup_file_dialog();
}

void ProjectManager::_new_project() {
	npdialog->set_mode(ProjectDialog::MODE_NEW);
	npdialog->show_dialog();
}

void ProjectManager::_import_project() {
	npdialog->set_mode(ProjectDialog::MODE_IMPORT);
	npdialog->show_dialog();
}

void ProjectManager::_rename_project() {
	const HashSet<String> &selected_list = _project_list->get_selected_project_keys();

	if (selected_list.size() == 0) {
		return;
	}

	for (const String &E : selected_list) {
		npdialog->set_project_path(E);
		npdialog->set_mode(ProjectDialog::MODE_RENAME);
		npdialog->show_dialog();
	}
}

void ProjectManager::_erase_project_confirm() {
	_project_list->erase_selected_projects(delete_project_contents->is_pressed());
	_update_project_buttons();
}

void ProjectManager::_erase_missing_projects_confirm() {
	_project_list->erase_missing_projects();
	_update_project_buttons();
}

void ProjectManager::_erase_project() {
	const HashSet<String> &selected_list = _project_list->get_selected_project_keys();

	if (selected_list.size() == 0) {
		return;
	}

	String confirm_message;
	if (selected_list.size() >= 2) {
		confirm_message = vformat(TTR("Remove %d projects from the list?"), selected_list.size());
	} else {
		confirm_message = TTR("Remove this project from the list?");
	}

	erase_ask_label->set_text(confirm_message);
	delete_project_contents->set_pressed(false);
	erase_ask->popup_centered();
}

void ProjectManager::_erase_missing_projects() {
	erase_missing_ask->set_text(TTR("Remove all missing projects from the list?\nThe project folders' contents won't be modified."));
	erase_missing_ask->popup_centered();
}

void ProjectManager::_show_about() {
	about->popup_centered(Size2(780, 500) * EDSCALE);
}

void ProjectManager::_language_selected(int p_id) {
	String lang = language_btn->get_item_metadata(p_id);
	EditorSettings::get_singleton()->set("interface/editor/editor_language", lang);

	language_restart_ask->set_text(TTR("Language changed.\nThe interface will update after restarting the editor or project manager."));
	language_restart_ask->popup_centered();
}

void ProjectManager::_restart_confirm() {
	List<String> args = OS::get_singleton()->get_cmdline_args();
	Error err = OS::get_singleton()->create_instance(args);
	ERR_FAIL_COND(err);

	_dim_window();
	get_tree()->quit();
}

void ProjectManager::_install_project(const String &p_zip_path, const String &p_title) {
	npdialog->set_mode(ProjectDialog::MODE_INSTALL);
	npdialog->set_zip_path(p_zip_path);
	npdialog->set_zip_title(p_title);
	npdialog->show_dialog();
}

void ProjectManager::_files_dropped(PackedStringArray p_files) {
	if (p_files.size() == 1 && p_files[0].ends_with(".zip")) {
		const String file = p_files[0].get_file();
		_install_project(p_files[0], file.substr(0, file.length() - 4).capitalize());
		return;
	}
	HashSet<String> folders_set;
	Ref<DirAccess> da = DirAccess::create(DirAccess::ACCESS_FILESYSTEM);
	for (int i = 0; i < p_files.size(); i++) {
		String file = p_files[i];
		folders_set.insert(da->dir_exists(file) ? file : file.get_base_dir());
	}
	if (folders_set.size() > 0) {
		PackedStringArray folders;
		for (const String &E : folders_set) {
			folders.push_back(E);
		}

		bool confirm = true;
		if (folders.size() == 1) {
			Ref<DirAccess> dir = DirAccess::create(DirAccess::ACCESS_FILESYSTEM);
			if (dir->change_dir(folders[0]) == OK) {
				dir->list_dir_begin();
				String file = dir->get_next();
				while (confirm && !file.is_empty()) {
					if (!dir->current_is_dir() && file.ends_with("project.godot")) {
						confirm = false;
					}
					file = dir->get_next();
				}
				dir->list_dir_end();
			}
		}
		if (confirm) {
			multi_scan_ask->get_ok_button()->disconnect("pressed", callable_mp(this, &ProjectManager::_scan_multiple_folders));
			multi_scan_ask->get_ok_button()->connect("pressed", callable_mp(this, &ProjectManager::_scan_multiple_folders).bind(folders));
			multi_scan_ask->set_text(
					vformat(TTR("Are you sure to scan %s folders for existing Godot projects?\nThis could take a while."), folders.size()));
			multi_scan_ask->popup_centered();
		} else {
			_scan_multiple_folders(folders);
		}
	}
}

void ProjectManager::_scan_multiple_folders(PackedStringArray p_files) {
	for (int i = 0; i < p_files.size(); i++) {
		_scan_begin(p_files.get(i));
	}
}

void ProjectManager::_on_order_option_changed(int p_idx) {
	if (is_inside_tree()) {
		_project_list->set_order_option(p_idx);
	}
}

void ProjectManager::_on_tab_changed(int p_tab) {
#ifndef ANDROID_ENABLED
	if (p_tab == 0) { // Projects
		// Automatically grab focus when the user moves from the Templates tab
		// back to the Projects tab.
		search_box->grab_focus();
	}

	// The Templates tab's search field is focused on display in the asset
	// library editor plugin code.
#endif
}

void ProjectManager::_on_search_term_changed(const String &p_term) {
	_project_list->set_search_term(p_term);
	_project_list->sort_projects();

	// Select the first visible project in the list.
	// This makes it possible to open a project without ever touching the mouse,
	// as the search field is automatically focused on startup.
	_project_list->select_first_visible_project();
	_update_project_buttons();
}

void ProjectManager::_bind_methods() {
	ClassDB::bind_method("_update_project_buttons", &ProjectManager::_update_project_buttons);
	ClassDB::bind_method("_version_button_pressed", &ProjectManager::_version_button_pressed);
}

void ProjectManager::_open_asset_library() {
	asset_library->disable_community_support();
	tabs->set_current_tab(1);
}

void ProjectManager::_version_button_pressed() {
	DisplayServer::get_singleton()->clipboard_set(version_btn->get_text());
}

ProjectManager::ProjectManager() {
	singleton = this;

	// load settings
	if (!EditorSettings::get_singleton()) {
		EditorSettings::create();
	}

	// Turn off some servers we aren't going to be using in the Project Manager.
	NavigationServer3D::get_singleton()->set_active(false);
	PhysicsServer3D::get_singleton()->set_active(false);
	PhysicsServer2D::get_singleton()->set_active(false);

	EditorSettings::get_singleton()->set_optimize_save(false); //just write settings as they came

	{
		int display_scale = EditorSettings::get_singleton()->get("interface/editor/display_scale");

		switch (display_scale) {
			case 0:
				// Try applying a suitable display scale automatically.
				editor_set_scale(EditorSettings::get_singleton()->get_auto_display_scale());
				break;
			case 1:
				editor_set_scale(0.75);
				break;
			case 2:
				editor_set_scale(1.0);
				break;
			case 3:
				editor_set_scale(1.25);
				break;
			case 4:
				editor_set_scale(1.5);
				break;
			case 5:
				editor_set_scale(1.75);
				break;
			case 6:
				editor_set_scale(2.0);
				break;
			default:
				editor_set_scale(EditorSettings::get_singleton()->get("interface/editor/custom_display_scale"));
				break;
		}
		EditorFileDialog::get_icon_func = &ProjectManager::_file_dialog_get_icon;
	}

	// TRANSLATORS: This refers to the application where users manage their Godot projects.
	DisplayServer::get_singleton()->window_set_title(VERSION_NAME + String(" - ") + TTR("Project Manager", "Application"));

	EditorFileDialog::set_default_show_hidden_files(EditorSettings::get_singleton()->get("filesystem/file_dialog/show_hidden_files"));

	set_anchors_and_offsets_preset(Control::PRESET_FULL_RECT);
	set_theme(create_custom_theme());

	set_anchors_and_offsets_preset(Control::PRESET_FULL_RECT);

	Panel *panel = memnew(Panel);
	add_child(panel);
	panel->set_anchors_and_offsets_preset(Control::PRESET_FULL_RECT);
	panel->add_theme_style_override("panel", get_theme_stylebox(SNAME("Background"), SNAME("EditorStyles")));

	VBoxContainer *vb = memnew(VBoxContainer);
	panel->add_child(vb);
	vb->set_anchors_and_offsets_preset(Control::PRESET_FULL_RECT, Control::PRESET_MODE_MINSIZE, 8 * EDSCALE);

	Control *center_box = memnew(Control);
	center_box->set_v_size_flags(Control::SIZE_EXPAND_FILL);
	vb->add_child(center_box);

	tabs = memnew(TabContainer);
	center_box->add_child(tabs);
	tabs->set_anchors_and_offsets_preset(Control::PRESET_FULL_RECT);
	tabs->connect("tab_changed", callable_mp(this, &ProjectManager::_on_tab_changed));

	local_projects_hb = memnew(HBoxContainer);
	local_projects_hb->set_name(TTR("Local Projects"));
	tabs->add_child(local_projects_hb);

	{
		// Projects + search bar
		VBoxContainer *search_tree_vb = memnew(VBoxContainer);
		local_projects_hb->add_child(search_tree_vb);
		search_tree_vb->set_h_size_flags(Control::SIZE_EXPAND_FILL);

		HBoxContainer *hb = memnew(HBoxContainer);
		hb->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		search_tree_vb->add_child(hb);

		search_box = memnew(LineEdit);
		search_box->set_placeholder(TTR("Filter Projects"));
		search_box->set_tooltip_text(TTR("This field filters projects by name and last path component.\nTo filter projects by name and full path, the query must contain at least one `/` character."));
		search_box->connect("text_changed", callable_mp(this, &ProjectManager::_on_search_term_changed));
		search_box->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		hb->add_child(search_box);

		loading_label = memnew(Label(TTR("Loading, please wait...")));
		loading_label->add_theme_font_override("font", get_theme_font(SNAME("bold"), SNAME("EditorFonts")));
		loading_label->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		hb->add_child(loading_label);
		// The loading label is shown later.
		loading_label->hide();

		Label *sort_label = memnew(Label);
		sort_label->set_text(TTR("Sort:"));
		hb->add_child(sort_label);

		filter_option = memnew(OptionButton);
		filter_option->set_clip_text(true);
		filter_option->set_h_size_flags(Control::SIZE_EXPAND_FILL);
		filter_option->connect("item_selected", callable_mp(this, &ProjectManager::_on_order_option_changed));
		hb->add_child(filter_option);

		Vector<String> sort_filter_titles;
		sort_filter_titles.push_back(TTR("Last Edited"));
		sort_filter_titles.push_back(TTR("Name"));
		sort_filter_titles.push_back(TTR("Path"));

		for (int i = 0; i < sort_filter_titles.size(); i++) {
			filter_option->add_item(sort_filter_titles[i]);
		}

		PanelContainer *pc = memnew(PanelContainer);
		pc->add_theme_style_override("panel", get_theme_stylebox(SNAME("panel"), SNAME("Tree")));
		pc->set_v_size_flags(Control::SIZE_EXPAND_FILL);
		search_tree_vb->add_child(pc);

		_project_list = memnew(ProjectList);
		_project_list->connect(ProjectList::SIGNAL_SELECTION_CHANGED, callable_mp(this, &ProjectManager::_update_project_buttons));
		_project_list->connect(ProjectList::SIGNAL_PROJECT_ASK_OPEN, callable_mp(this, &ProjectManager::_open_selected_projects_ask));
		_project_list->set_horizontal_scroll_mode(ScrollContainer::SCROLL_MODE_DISABLED);
		pc->add_child(_project_list);
	}

	{
		// Project tab side bar
		VBoxContainer *tree_vb = memnew(VBoxContainer);
		tree_vb->set_custom_minimum_size(Size2(120, 120));
		local_projects_hb->add_child(tree_vb);

		const int btn_h_separation = int(6 * EDSCALE);

		create_btn = memnew(Button);
		create_btn->set_text(TTR("New Project"));
		create_btn->add_theme_constant_override("h_separation", btn_h_separation);
		create_btn->set_shortcut(ED_SHORTCUT("project_manager/new_project", TTR("New Project"), KeyModifierMask::CMD_OR_CTRL | Key::N));
		create_btn->connect("pressed", callable_mp(this, &ProjectManager::_new_project));
		tree_vb->add_child(create_btn);

		import_btn = memnew(Button);
		import_btn->set_text(TTR("Import"));
		import_btn->add_theme_constant_override("h_separation", btn_h_separation);
		import_btn->set_shortcut(ED_SHORTCUT("project_manager/import_project", TTR("Import Project"), KeyModifierMask::CMD_OR_CTRL | Key::I));
		import_btn->connect("pressed", callable_mp(this, &ProjectManager::_import_project));
		tree_vb->add_child(import_btn);

		scan_btn = memnew(Button);
		scan_btn->set_text(TTR("Scan"));
		scan_btn->add_theme_constant_override("h_separation", btn_h_separation);
		scan_btn->set_shortcut(ED_SHORTCUT("project_manager/scan_projects", TTR("Scan Projects"), KeyModifierMask::CMD_OR_CTRL | Key::S));
		scan_btn->connect("pressed", callable_mp(this, &ProjectManager::_scan_projects));
		tree_vb->add_child(scan_btn);

		tree_vb->add_child(memnew(HSeparator));

		open_btn = memnew(Button);
		open_btn->set_text(TTR("Edit"));
		open_btn->add_theme_constant_override("h_separation", btn_h_separation);
		open_btn->set_shortcut(ED_SHORTCUT("project_manager/edit_project", TTR("Edit Project"), KeyModifierMask::CMD_OR_CTRL | Key::E));
		open_btn->connect("pressed", callable_mp(this, &ProjectManager::_open_selected_projects_ask));
		tree_vb->add_child(open_btn);

		run_btn = memnew(Button);
		run_btn->set_text(TTR("Run"));
		run_btn->add_theme_constant_override("h_separation", btn_h_separation);
		run_btn->set_shortcut(ED_SHORTCUT("project_manager/run_project", TTR("Run Project"), KeyModifierMask::CMD_OR_CTRL | Key::R));
		run_btn->connect("pressed", callable_mp(this, &ProjectManager::_run_project));
		tree_vb->add_child(run_btn);

		rename_btn = memnew(Button);
		rename_btn->set_text(TTR("Rename"));
		rename_btn->add_theme_constant_override("h_separation", btn_h_separation);
		// The F2 shortcut isn't overridden with Enter on macOS as Enter is already used to edit a project.
		rename_btn->set_shortcut(ED_SHORTCUT("project_manager/rename_project", TTR("Rename Project"), Key::F2));
		rename_btn->connect("pressed", callable_mp(this, &ProjectManager::_rename_project));
		tree_vb->add_child(rename_btn);

		erase_btn = memnew(Button);
		erase_btn->set_text(TTR("Remove"));
		erase_btn->add_theme_constant_override("h_separation", btn_h_separation);
		erase_btn->set_shortcut(ED_SHORTCUT("project_manager/remove_project", TTR("Remove Project"), Key::KEY_DELETE));
		erase_btn->connect("pressed", callable_mp(this, &ProjectManager::_erase_project));
		tree_vb->add_child(erase_btn);

		erase_missing_btn = memnew(Button);
		erase_missing_btn->set_text(TTR("Remove Missing"));
		erase_missing_btn->add_theme_constant_override("h_separation", btn_h_separation);
		erase_missing_btn->connect("pressed", callable_mp(this, &ProjectManager::_erase_missing_projects));
		tree_vb->add_child(erase_missing_btn);

		tree_vb->add_spacer();

		about_btn = memnew(Button);
		about_btn->set_text(TTR("About"));
		about_btn->connect("pressed", callable_mp(this, &ProjectManager::_show_about));
		tree_vb->add_child(about_btn);
	}

	{
		// Version info and language options
		settings_hb = memnew(HBoxContainer);
		settings_hb->set_alignment(BoxContainer::ALIGNMENT_END);
		settings_hb->set_h_grow_direction(Control::GROW_DIRECTION_BEGIN);
		settings_hb->set_anchors_and_offsets_preset(Control::PRESET_TOP_RIGHT);

		// A VBoxContainer that contains a dummy Control node to adjust the LinkButton's vertical position.
		VBoxContainer *spacer_vb = memnew(VBoxContainer);
		settings_hb->add_child(spacer_vb);

		Control *v_spacer = memnew(Control);
		spacer_vb->add_child(v_spacer);

		version_btn = memnew(LinkButton);
		String hash = String(VERSION_HASH);
		if (hash.length() != 0) {
			hash = " " + vformat("[%s]", hash.left(9));
		}
		version_btn->set_text("v" VERSION_FULL_BUILD + hash);
		// Fade the version label to be less prominent, but still readable.
		version_btn->set_self_modulate(Color(1, 1, 1, 0.6));
		version_btn->set_underline_mode(LinkButton::UNDERLINE_MODE_ON_HOVER);
		version_btn->set_tooltip_text(TTR("Click to copy."));
		version_btn->connect("pressed", callable_mp(this, &ProjectManager::_version_button_pressed));
		spacer_vb->add_child(version_btn);

		// Add a small horizontal spacer between the version and language buttons
		// to distinguish them.
		Control *h_spacer = memnew(Control);
		settings_hb->add_child(h_spacer);

		language_btn = memnew(OptionButton);
		language_btn->set_icon(get_theme_icon(SNAME("Environment"), SNAME("EditorIcons")));
		language_btn->set_focus_mode(Control::FOCUS_NONE);
		language_btn->set_fit_to_longest_item(false);
		language_btn->set_flat(true);
		language_btn->connect("item_selected", callable_mp(this, &ProjectManager::_language_selected));
#ifdef ANDROID_ENABLED
		// The language selection dropdown doesn't work on Android (as the setting isn't saved), see GH-60353.
		// Also, the dropdown it spawns is very tall and can't be scrolled without a hardware mouse.
		// Hiding the language selection dropdown also leaves more space for the version label to display.
		language_btn->hide();
#endif

		Vector<String> editor_languages;
		List<PropertyInfo> editor_settings_properties;
		EditorSettings::get_singleton()->get_property_list(&editor_settings_properties);
		for (const PropertyInfo &pi : editor_settings_properties) {
			if (pi.name == "interface/editor/editor_language") {
				editor_languages = pi.hint_string.split(",");
				break;
			}
		}

		String current_lang = EditorSettings::get_singleton()->get("interface/editor/editor_language");
		language_btn->set_text(current_lang);

		for (int i = 0; i < editor_languages.size(); i++) {
			String lang = editor_languages[i];
			String lang_name = TranslationServer::get_singleton()->get_locale_name(lang);
			language_btn->add_item(vformat("[%s] %s", lang, lang_name), i);
			language_btn->set_item_metadata(i, lang);
			if (current_lang == lang) {
				language_btn->select(i);
			}
		}

		settings_hb->add_child(language_btn);
		center_box->add_child(settings_hb);
	}

	if (AssetLibraryEditorPlugin::is_available()) {
		asset_library = memnew(EditorAssetLibrary(true));
		asset_library->set_name(TTR("Asset Library Projects"));
		tabs->add_child(asset_library);
		asset_library->connect("install_asset", callable_mp(this, &ProjectManager::_install_project));
	} else {
		print_verbose("Asset Library not available (due to using Web editor, or SSL support disabled).");
	}

	{
		// Dialogs
		language_restart_ask = memnew(ConfirmationDialog);
		language_restart_ask->set_ok_button_text(TTR("Restart Now"));
		language_restart_ask->get_ok_button()->connect("pressed", callable_mp(this, &ProjectManager::_restart_confirm));
		language_restart_ask->set_cancel_button_text(TTR("Continue"));
		add_child(language_restart_ask);

		scan_dir = memnew(EditorFileDialog);
		scan_dir->set_previews_enabled(false);
		scan_dir->set_access(EditorFileDialog::ACCESS_FILESYSTEM);
		scan_dir->set_file_mode(EditorFileDialog::FILE_MODE_OPEN_DIR);
		scan_dir->set_title(TTR("Select a Folder to Scan")); // must be after mode or it's overridden
		scan_dir->set_current_dir(EditorSettings::get_singleton()->get("filesystem/directories/default_project_path"));
		add_child(scan_dir);
		scan_dir->connect("dir_selected", callable_mp(this, &ProjectManager::_scan_begin));

		erase_missing_ask = memnew(ConfirmationDialog);
		erase_missing_ask->set_ok_button_text(TTR("Remove All"));
		erase_missing_ask->get_ok_button()->connect("pressed", callable_mp(this, &ProjectManager::_erase_missing_projects_confirm));
		add_child(erase_missing_ask);

		erase_ask = memnew(ConfirmationDialog);
		erase_ask->set_ok_button_text(TTR("Remove"));
		erase_ask->get_ok_button()->connect("pressed", callable_mp(this, &ProjectManager::_erase_project_confirm));
		add_child(erase_ask);

		VBoxContainer *erase_ask_vb = memnew(VBoxContainer);
		erase_ask->add_child(erase_ask_vb);

		erase_ask_label = memnew(Label);
		erase_ask_vb->add_child(erase_ask_label);

		delete_project_contents = memnew(CheckBox);
		delete_project_contents->set_text(TTR("Also delete project contents (no undo!)"));
		erase_ask_vb->add_child(delete_project_contents);

		multi_open_ask = memnew(ConfirmationDialog);
		multi_open_ask->set_ok_button_text(TTR("Edit"));
		multi_open_ask->get_ok_button()->connect("pressed", callable_mp(this, &ProjectManager::_open_selected_projects));
		add_child(multi_open_ask);

		multi_run_ask = memnew(ConfirmationDialog);
		multi_run_ask->set_ok_button_text(TTR("Run"));
		multi_run_ask->get_ok_button()->connect("pressed", callable_mp(this, &ProjectManager::_run_project_confirm));
		add_child(multi_run_ask);

		multi_scan_ask = memnew(ConfirmationDialog);
		multi_scan_ask->set_ok_button_text(TTR("Scan"));
		add_child(multi_scan_ask);

		ask_update_settings = memnew(ConfirmationDialog);
		ask_update_settings->get_ok_button()->connect("pressed", callable_mp(this, &ProjectManager::_confirm_update_settings));
		full_convert_button = ask_update_settings->add_button("Perform Full Project Conversion", false);
		add_child(ask_update_settings);

		ask_full_convert_dialog = memnew(ConfirmationDialog);
		ask_full_convert_dialog->set_autowrap(true);
		ask_full_convert_dialog->set_text(TTR(R"(This option will perform full project conversion, updating scenes and scripts from Godot 3.x to work in Godot 4.0. Note that this is a best-effort conversion, i.e. it makes upgrading the project easier, but it will not open out-of-the-box and will still require manual adjustments.

IMPORTANT: Make sure to backup your project before converting, as this operation makes it impossible to open in older versions of Godot.)"));
		ask_full_convert_dialog->connect("confirmed", callable_mp(this, &ProjectManager::_perform_full_project_conversion));
		add_child(ask_full_convert_dialog);

		full_convert_button->connect("pressed", callable_mp((Window *)ask_update_settings, &ConfirmationDialog::hide));
		full_convert_button->connect("pressed", callable_mp((Window *)ask_full_convert_dialog, &ConfirmationDialog::popup_centered_ratio).bind(0.2));

		npdialog = memnew(ProjectDialog);
		npdialog->connect("projects_updated", callable_mp(this, &ProjectManager::_on_projects_updated));
		npdialog->connect("project_created", callable_mp(this, &ProjectManager::_on_project_created));
		add_child(npdialog);

		run_error_diag = memnew(AcceptDialog);
		run_error_diag->set_title(TTR("Can't run project"));
		add_child(run_error_diag);

		dialog_error = memnew(AcceptDialog);
		add_child(dialog_error);

		if (asset_library) {
			open_templates = memnew(ConfirmationDialog);
			open_templates->set_text(TTR("You currently don't have any projects.\nWould you like to explore official example projects in the Asset Library?"));
			open_templates->set_ok_button_text(TTR("Open Asset Library"));
			open_templates->connect("confirmed", callable_mp(this, &ProjectManager::_open_asset_library));
			add_child(open_templates);
		}

		about = memnew(EditorAbout);
		add_child(about);

		_build_icon_type_cache(get_theme());
	}

	_project_list->migrate_config();
	_load_recent_projects();

	Ref<DirAccess> dir_access = DirAccess::create(DirAccess::AccessType::ACCESS_FILESYSTEM);

	String default_project_path = EditorSettings::get_singleton()->get("filesystem/directories/default_project_path");
	if (!dir_access->dir_exists(default_project_path)) {
		Error error = dir_access->make_dir_recursive(default_project_path);
		if (error != OK) {
			ERR_PRINT("Could not create default project directory at: " + default_project_path);
		}
	}

	String autoscan_path = EditorSettings::get_singleton()->get("filesystem/directories/autoscan_project_path");
	if (!autoscan_path.is_empty()) {
		if (dir_access->dir_exists(autoscan_path)) {
			_scan_begin(autoscan_path);
		} else {
			Error error = dir_access->make_dir_recursive(autoscan_path);
			if (error != OK) {
				ERR_PRINT("Could not create project autoscan directory at: " + autoscan_path);
			}
		}
	}

	SceneTree::get_singleton()->get_root()->connect("files_dropped", callable_mp(this, &ProjectManager::_files_dropped));

	// Define a minimum window size to prevent UI elements from overlapping or being cut off.
	DisplayServer::get_singleton()->window_set_min_size(Size2(520, 350) * EDSCALE);

	// Resize the bootsplash window based on Editor display scale EDSCALE.
	float scale_factor = MAX(1, EDSCALE);
	if (scale_factor > 1.0) {
		Vector2i window_size = DisplayServer::get_singleton()->window_get_size();
		Vector2i screen_size = DisplayServer::get_singleton()->screen_get_size();
		Vector2i screen_position = DisplayServer::get_singleton()->screen_get_position();

		window_size *= scale_factor;

		DisplayServer::get_singleton()->window_set_size(window_size);
		if (screen_size != Vector2i()) {
			Vector2i window_position;
			window_position.x = screen_position.x + (screen_size.x - window_size.x) / 2;
			window_position.y = screen_position.y + (screen_size.y - window_size.y) / 2;
			DisplayServer::get_singleton()->window_set_position(window_position);
		}
	}

	OS::get_singleton()->set_low_processor_usage_mode(true);
}

ProjectManager::~ProjectManager() {
	singleton = nullptr;
	if (EditorSettings::get_singleton()) {
		EditorSettings::destroy();
	}
}
